# Oinone 异常处理规范

## 1. 核心原则

1. **统一使用 `PamirsException`**：禁止使用 `RuntimeException`
2. **枚举定义错误码和消息**：所有业务异常通过枚举定义
3. **参数化消息**：使用构造函数参数格式化消息，禁止写死前缀
4. **结构化上下文**：使用 `setExtendObject()` 携带复杂上下文信息
5. **分层处理**：Service 层抛出异常，Action 层不处理异常

---

## 2. 异常枚举定义规范

### 2.1 枚举命名规则

```java
// 格式：{模块}_{实体}_{错误类型}
PRODUCT_NULL              // 产品为空
PRODUCT_CODE_EXISTS       // 产品编码已存在
QUOTATION_STATUS_INVALID  // 报价单状态无效
TICKET_TIMEOUT           // 工单超时
```

### 2.2 错误码分配规则

| 模块 | 错误码范围 | 说明 |
|------|-----------|------|
| 产品 | 30001001-30001999 | 产品相关异常 |
| 报价单 | 30002001-30002999 | 报价单相关异常 |
| 合同 | 30003001-30003999 | 合同相关异常 |
| 工单 | 30004001-30004999 | 工单相关异常 |
| 营销活动 | 30005001-30005999 | 营销活动相关异常 |
| 销售目标 | 30006001-30006999 | 销售目标相关异常 |
| 销售预测 | 30007001-30007999 | 销售预测相关异常 |
| 其他 | 30008001-30009999 | 其他异常 |

### 2.3 消息模板规范

```java
// 简单消息（无参数）
PRODUCT_NULL(ERROR_TYPE.BIZ_ERROR, 30001001, "产品不能为空"),

// 单参数消息（使用 {0} 占位符）
PRODUCT_NOT_FOUND(ERROR_TYPE.BIZ_ERROR, 30001005, "产品不存在，ID: {0}"),
PRODUCT_CODE_EXISTS(ERROR_TYPE.BIZ_ERROR, 30001003, "产品编码已存在: {0}"),

// 多参数消息（使用 {0}, {1}, {2} 占位符）
SALES_TARGET_DATE_INVALID(ERROR_TYPE.BIZ_ERROR, 30006006, "开始日期不能晚于结束日期，开始: {0}, 结束: {1}"),
```

### 2.4 完整枚举示例

```java
package pro.shushi.oinone.{module}.api.error;

import pro.shushi.pamirs.meta.common.enmu.ExpBaseEnum;
import pro.shushi.pamirs.meta.annotation.Errors;

@Errors(displayName = "{module}模块错误枚举")
public enum {module}ExceptionEnum implements ExpBaseEnum  {
    
    // ========== 产品相关 (1001-1999) ==========
    PRODUCT_NULL(ERROR_TYPE.BIZ_ERROR, 30001001, "产品不能为空"),
    PRODUCT_CODE_NULL(ERROR_TYPE.BIZ_ERROR, 30001002, "产品编码不能为空"),
    PRODUCT_CODE_EXISTS(ERROR_TYPE.BIZ_ERROR, 30001003, "产品编码已存在: {0}"),
    PRODUCT_NAME_NULL(ERROR_TYPE.BIZ_ERROR, 30001004, "产品名称不能为空"),
    PRODUCT_NOT_FOUND(ERROR_TYPE.BIZ_ERROR, 30001005, "产品不存在，ID: {0}"),
    PRODUCT_CODE_CANNOT_MODIFY(ERROR_TYPE.BIZ_ERROR, 30001006, "产品编码不能修改"),
    
    // ========== 报价单相关 (2001-2999) ==========
    QUOTATION_NULL(ERROR_TYPE.BIZ_ERROR, 30002001, "报价单不能为空"),
    QUOTATION_NO_NULL(ERROR_TYPE.BIZ_ERROR, 30002002, "报价单号不能为空"),
    QUOTATION_NO_EXISTS(ERROR_TYPE.BIZ_ERROR, 30002003, "报价单号已存在: {0}"),
    QUOTATION_NOT_FOUND(ERROR_TYPE.BIZ_ERROR, 30002004, "报价单不存在，ID: {0}"),
    QUOTATION_CUSTOMER_NULL(ERROR_TYPE.BIZ_ERROR, 30002005, "客户不能为空"),
    QUOTATION_STATUS_NOT_PENDING(ERROR_TYPE.BIZ_ERROR, 30002006, "报价单状态不是待审批，当前状态: {0}"),
    QUOTATION_NOT_APPROVED(ERROR_TYPE.BIZ_ERROR, 30002007, "报价单未审批，不能签署"),
    
    // ========== 合同相关 (3001-3999) ==========
    CONTRACT_NULL(ERROR_TYPE.BIZ_ERROR, 30003001, "合同不能为空"),
    CONTRACT_NO_NULL(ERROR_TYPE.BIZ_ERROR, 30003002, "合同号不能为空"),
    CONTRACT_NO_EXISTS(ERROR_TYPE.BIZ_ERROR, 30003003, "合同号已存在: {0}"),
    CONTRACT_NOT_FOUND(ERROR_TYPE.BIZ_ERROR, 30003004, "合同不存在，ID: {0}"),
    CONTRACT_CUSTOMER_NULL(ERROR_TYPE.BIZ_ERROR, 30003005, "客户不能为空"),
    CONTRACT_STATUS_NOT_PENDING(ERROR_TYPE.BIZ_ERROR, 30003006, "合同状态不是待审批，当前状态: {0}"),
    CONTRACT_NOT_APPROVED(ERROR_TYPE.BIZ_ERROR, 30003007, "合同未审批，不能签署"),
    
    // ========== 工单相关 (4001-4999) ==========
    TICKET_NULL(ERROR_TYPE.BIZ_ERROR, 30004001, "工单不能为空"),
    TICKET_NO_NULL(ERROR_TYPE.BIZ_ERROR, 30004002, "工单号不能为空"),
    TICKET_NOT_FOUND(ERROR_TYPE.BIZ_ERROR, 30004003, "工单不存在，ID: {0}"),
    TICKET_CUSTOMER_NULL(ERROR_TYPE.BIZ_ERROR, 30004004, "客户不能为空"),
    TICKET_HANDLER_NULL(ERROR_TYPE.BIZ_ERROR, 30004005, "处理人不能为空"),
    TICKET_NULL(ERROR_TYPE.BIZ_ERROR,30004001, "工单不能为空"),
    TICKET_NO_NULL(ERROR_TYPE.BIZ_ERROR,30004002, "工单号不能为空"),
    TICKET_NOT_FOUND(ERROR_TYPE.BIZ_ERROR,30004003, "工单不存在，ID: {0}"),
    TICKET_CUSTOMER_NULL(ERROR_TYPE.BIZ_ERROR,30004004, "客户不能为空"),
    TICKET_HANDLER_NULL(ERROR_TYPE.BIZ_ERROR,30004005, "处理人不能为空"),
    TICKET_STATUS_NOT_PENDING(ERROR_TYPE.BIZ_ERROR,30004006, "工单状态不是待受理，当前状态: {0}"),
    TICKET_ALREADY_CLOSED(ERROR_TYPE.BIZ_ERROR,30004007, "工单已关闭: {0}"),
    TICKET_NO_HANDLER(ERROR_TYPE.BIZ_ERROR,30004008, "工单未分配处理人"),
    TICKET_PROCESS_CONTENT_NULL(ERROR_TYPE.BIZ_ERROR,30004009, "处理内容不能为空"),
    
    // ========== 营销活动相关 (5001-5999) ==========
    CAMPAIGN_NULL(ERROR_TYPE.BIZ_ERROR,30005001, "营销活动不能为空"),
    CAMPAIGN_NO_NULL(ERROR_TYPE.BIZ_ERROR,30005002, "营销活动编号不能为空"),
    CAMPAIGN_NAME_NULL(ERROR_TYPE.BIZ_ERROR,30005003, "营销活动名称不能为空"),
    CAMPAIGN_NOT_FOUND(ERROR_TYPE.BIZ_ERROR,30005004, "营销活动不存在，ID: {0}"),
    CAMPAIGN_STATUS_NOT_STARTED(ERROR_TYPE.BIZ_ERROR,30005005, "营销活动状态不是未开始，当前状态: {0}"),
    CAMPAIGN_STATUS_NOT_IN_PROGRESS(ERROR_TYPE.BIZ_ERROR,30005006, "营销活动状态不是进行中，当前状态: {0}"),
    CAMPAIGN_REACH_NO_NULL(ERROR_TYPE.BIZ_ERROR,30005007, "触达编号不能为空"),
    CAMPAIGN_REACH_TARGET_NULL(ERROR_TYPE.BIZ_ERROR,30005008, "触达目标不能为空"),
    CAMPAIGN_REACH_NOT_FOUND(ERROR_TYPE.BIZ_ERROR,30005009, "触达不存在，ID: {0}"),
    CAMPAIGN_REACH_RESULT_TYPE_NULL(ERROR_TYPE.BIZ_ERROR,30005010, "结果类型不能为空"),
    
    // ========== 销售目标相关 (6001-6999) ==========
    SALES_TARGET_NULL(ERROR_TYPE.BIZ_ERROR,30006001, "销售目标不能为空"),
    SALES_TARGET_NO_NULL(ERROR_TYPE.BIZ_ERROR,30006002, "目标编号不能为空"),
    SALES_TARGET_AMOUNT_INVALID(ERROR_TYPE.BIZ_ERROR,30006003, "目标金额必须大于0，当前金额: {0}"),
    SALES_TARGET_START_DATE_NULL(ERROR_TYPE.BIZ_ERROR,30006004, "开始日期不能为空"),
    SALES_TARGET_END_DATE_NULL(ERROR_TYPE.BIZ_ERROR,30006005, "结束日期不能为空"),
    SALES_TARGET_DATE_INVALID(ERROR_TYPE.BIZ_ERROR,30006006, "开始日期不能晚于结束日期，开始: {0}, 结束: {1}"),
    SALES_TARGET_NOT_FOUND(ERROR_TYPE.BIZ_ERROR,30006007, "销售目标不存在，ID: {0}");

    private final ERROR_TYPE type;
    private final int code;
    private final String msg;
    
    {module}ExceptionEnum(ERROR_TYPE type, int code, String msg) {
        this.type = type;
        this.code = code;
        this.msg = msg;
    }

    @Override
    public ERROR_TYPE type() {
        return type;
    }

    
    @Override
    public int code() {
        return code;
    }
    
    @Override
    public String msg() {
        return msg;
    }
    
    @Override
    public ERROR_TYPE type() {
        return type;
    }
}
```

---

## 3. 异常抛出规范

### 3.1 场景 1：简单校验失败（无参数）

**适用场景**：参数为 null、空值等简单校验

```java
// ❌ 错误写法
if (product == null) {
    throw new RuntimeException("产品不能为空");
}

// ✅ 正确写法
if (product == null) {
    throw PamirsException.construct({module}ExceptionEnum.PRODUCT_NULL).errThrow();
}
```

### 3.2 场景 2：需要告知具体值（单参数）

**适用场景**：需要告知具体的 ID、编码、名称等

```java
// ❌ 错误写法
if (product == null) {
    throw new RuntimeException("产品不存在，ID: " + productId);
}

// ❌ 错误写法（禁止写死前缀）
if (product == null) {
    throw PamirsException.construct({module}ExceptionEnum.PRODUCT_NOT_FOUND)
        .appendMsg("ID: " + productId)
        .errThrow();
}

// ✅ 正确写法（使用构造函数参数）
if (product == null) {
    throw PamirsException.construct(
        {module}ExceptionEnum.PRODUCT_NOT_FOUND, 
        productId
    ).errThrow();
}
```

### 3.3 场景 3：需要告知多个值（多参数）

**适用场景**：需要告知多个相关值

```java
// ❌ 错误写法
if (startDate.after(endDate)) {
    throw new RuntimeException("开始日期不能晚于结束日期，开始: " + startDate + ", 结束: " + endDate);
}

// ❌ 错误写法（禁止写死前缀）
if (startDate.after(endDate)) {
    throw PamirsException.construct({module}ExceptionEnum.SALES_TARGET_DATE_INVALID)
        .appendMsg("开始: " + startDate)
        .appendMsg("结束: " + endDate)
        .errThrow();
}

// ✅ 正确写法（使用构造函数参数）
if (startDate.after(endDate)) {
    throw PamirsException.construct(
        {module}ExceptionEnum.SALES_TARGET_DATE_INVALID,
        startDate,
        endDate
    ).errThrow();
}
```

### 3.4 场景 4：需要携带结构化上下文

**适用场景**：需要携带复杂的结构化信息，便于日志分析

```java
// ❌ 错误写法
if (!TicketStatusEnum.PENDING.equals(ticket.getStatus())) {
    throw new RuntimeException("工单状态不是待受理");
}

// ✅ 正确写法（使用 setExtendObject 携带上下文）
if (!TicketStatusEnum.PENDING.equals(ticket.getStatus())) {
    Map<String, Object> context = new HashMap<>();
    context.put("ticketNo", ticket.getTicketNo());
    context.put("currentStatus", ticket.getStatus().name());
    context.put("expectedStatus", "PENDING");
    
    throw PamirsException.construct(
        {module}ExceptionEnum.TICKET_STATUS_NOT_PENDING,
        ticket.getStatus().name()
    ).setExtendObject(context).errThrow();
}
```

### 3.5 场景 5：查询失败（带 ID）

```java
// ❌ 错误写法
Product product = new Product().setId(productId).queryById();
if (product == null) {
    throw new RuntimeException("产品不存在");
}

// ✅ 正确写法
Product product = new Product().setId(productId).queryById();
if (product == null) {
    throw PamirsException.construct(
        {module}ExceptionEnum.PRODUCT_NOT_FOUND,
        productId
    ).errThrow();
}
```

### 3.6 场景 6：业务规则校验失败

```java
// ❌ 错误写法
if (checkProductCodeExists(productCode)) {
    throw new RuntimeException("产品编码已存在: " + productCode);
}

// ✅ 正确写法
if (checkProductCodeExists(productCode)) {
    throw PamirsException.construct(
        {module}ExceptionEnum.PRODUCT_CODE_EXISTS,
        productCode
    ).errThrow();
}
```

---

## 4. 分层处理规范

### 4.1 Service 层

**职责**：业务逻辑校验，抛出异常

```java
@Service
@Fun(ProductService.FUN_NAMESPACE)
public class ProductServiceImpl implements ProductService {

    @Function
    @Override
    @PamirsTransactional
    public Product createProduct(Product product) {
        // 参数校验
        if (product == null) {
            throw PamirsException.construct({module}ExceptionEnum.PRODUCT_NULL).errThrow();
        }
        
        if (product.getProductCode() == null || product.getProductCode().isEmpty()) {
            throw PamirsException.construct({module}ExceptionEnum.PRODUCT_CODE_NULL).errThrow();
        }
        
        // 业务规则校验
        if (checkProductCodeExists(product.getProductCode())) {
            throw PamirsException.construct(
                {module}ExceptionEnum.PRODUCT_CODE_EXISTS,
                product.getProductCode()
            ).errThrow();
        }
        
        product.create();
        return product;
    }
}
```

### 4.2 Action 层

**职责**：协调 Service，不处理异常

```java
@Component
@Model.model(Product.MODEL_MODEL)
public class ProductAction {

    @Autowired
    private ProductService productService;

    @Action.Advanced(name = FunctionConstants.create, type = FunctionTypeEnum.CREATE, managed = true, invisible = ExpConstants.idValueExist, check = true)
    @Action(displayName = "创建产品", label = "创建产品", bindingType = ViewTypeEnum.FORM)
    @Function(name = FunctionConstants.create)
    @Function.fun(FunctionConstants.create)
    public Product createProduct(Product product) {
        // ✅ 直接调用 Service，不捕获异常
        return productService.createProduct(product);
    }
}
```

---

## 5. 异常处理最佳实践

### 5.1 参数校验顺序

```java
// ✅ 推荐：从外到内，从简单到复杂
public Product createProduct(Product product) {
    // 1. 对象校验
    if (product == null) {
        throw PamirsException.construct({module}ExceptionEnum.PRODUCT_NULL).errThrow();
    }
    
    // 2. 必填字段校验
    if (product.getProductCode() == null || product.getProductCode().isEmpty()) {
        throw PamirsException.construct({module}ExceptionEnum.PRODUCT_CODE_NULL).errThrow();    
    }
    
    if (product.getProductName() == null || product.getProductName().isEmpty()) {
        throw PamirsException.construct({module}ExceptionEnum.PRODUCT_NAME_NULL).errThrow();
    }
    
    // 3. 业务规则校验
    if (checkProductCodeExists(product.getProductCode())) {
        throw PamirsException.construct(
            {module}ExceptionEnum.PRODUCT_CODE_EXISTS,
            product.getProductCode()
        ).errThrow();
    }
    
    product.create();
    return product;
}
```

### 5.2 上下文信息设计

```java
// ✅ 推荐：上下文信息包含关键字段
Map<String, Object> context = new HashMap<>();
context.put("ticketNo", ticket.getTicketNo());        // 业务编号
context.put("currentStatus", ticket.getStatus().name()); // 当前状态
context.put("expectedStatus", "PENDING");              // 期望状态
context.put("handler", ticket.getHandler()?.getName()); // 处理人
context.put("createTime", ticket.getCreateTime());       // 创建时间

throw PamirsException.construct(
    {module}ExceptionEnum.TICKET_STATUS_NOT_PENDING,
    ticket.getStatus().name()
).setExtendObject(context).errThrow();
```

### 5.3 异常消息设计原则

```java
// ✅ 推荐：消息清晰、准确、完整
PRODUCT_NOT_FOUND(ERROR_TYPE.BIZ_ERROR,30001005, "产品不存在，ID: {0}"),                    // 清晰
PRODUCT_CODE_EXISTS(ERROR_TYPE.BIZ_ERROR,30001003, "产品编码已存在: {0}"),                      // 准确
SALES_TARGET_DATE_INVALID(ERROR_TYPE.BIZ_ERROR,30006006, "开始日期不能晚于结束日期，开始: {0}, 结束: {1}"), // 完整

// ❌ 避免：消息模糊、不准确、不完整
PRODUCT_NOT_FOUND(ERROR_TYPE.BIZ_ERROR,30001005, "产品不存在"),                    // 模糊，不知道哪个产品
PRODUCT_CODE_EXISTS(ERROR_TYPE.BIZ_ERROR,30001003, "编码已存在"),                        // 不准确，不知道什么编码
SALES_TARGET_DATE_INVALID(ERROR_TYPE.BIZ_ERROR,30006006, "日期无效"),              // 不完整，不知道哪个日期
```

---

## 6. 日志记录规范

```java
// ✅ 推荐：异常会自动记录，无需手动记录
throw PamirsException.construct(
    {module}ExceptionEnum.PRODUCT_NOT_FOUND,
    productId
).errThrow();

// ❌ 避免：手动记录日志后再抛出异常
log.error("产品不存在，ID: {}", productId);
throw PamirsException.construct(
    {module}ExceptionEnum.PRODUCT_NOT_FOUND,
    productId
).errThrow();
```

---

## 7. 快速参考

### 7.1 异常抛出模板

```java
// 无参数
throw PamirsException.construct({module}ExceptionEnum.XXX).errThrow();

// 单参数
throw PamirsException.construct({module}ExceptionEnum.XXX, value).errThrow();

// 多参数
throw PamirsException.construct({module}ExceptionEnum.XXX, value1, value2).errThrow();

// 带上下文
Map<String, Object> context = new HashMap<>();
context.put("key", value);
throw PamirsException.construct({module}ExceptionEnum.XXX, value)
    .setExtendObject(context)
    .errThrow();
```

### 7.2 禁止写法

```java
// ❌ 禁止：使用 RuntimeException
throw new RuntimeException("错误信息");

// ❌ 禁止：使用 appendMsg 写死前缀
throw PamirsException.construct({module}ExceptionEnum.XXX)
    .appendMsg("ID: " + id)
    .errThrow();

// ❌ 禁止：使用 setMsgDetail 写死前缀
throw PamirsException.construct({module}ExceptionEnum.XXX)
    .setMsgDetail("ID: " + id)
    .errThrow();

// ❌ 禁止：在 Action 层捕获异常
try {
    return service.method();
} catch (Exception e) {
    throw e; // 不要这样做
}
```

---

## 8. 总结

| 规则 | 说明 |
|------|------|
| ✅ 使用 `PamirsException` | 禁止使用 `RuntimeException` |
| ✅ 使用枚举定义错误码 | 所有错误码在 `OcrmExceptionEnum` 中定义 |
| ✅ 参数化消息 | 使用 `{0}`, `{1}` 占位符，禁止写死前缀 |
| ✅ 使用构造函数参数 | `construct(Enum, arg1, arg2)` |
| ✅ 使用 `setExtendObject()` | 携带结构化上下文 |
| ✅ Service 层抛出异常 | 业务校验失败时抛出异常 |
| ✅ Action 层不处理异常 | 直接调用 Service，不捕获异常 |
| ❌ 禁止 `appendMsg()` 写死前缀 | 使用构造函数参数代替 |
| ❌ 禁止 `setMsgDetail()` 写死前缀 | 使用构造函数参数代替 |
| ❌ 禁止在 Action 层捕获异常 | 让异常自然向上传播 |
