# Oinone 数据权限扩展规范

> **版本**: 2.0 | **基准**: Oinone 官方文档 | **适用**: 数据权限扩展开发

## 1. 概述

Oinone 框架提供了功能权限和字段权限的默认能力，但数据权限需要通过 Placeholder 扩展机制实现。本文档定义了 Oinone 数据权限扩展的规范和最佳实践。

### 1.1 权限类型说明

| 权限类型 | 说明 | 适用场景 |
|---------|------|---------|
| **功能权限** | 菜单权限、按钮权限 | 使用 Oinone 默认能力，无需扩展 |
| **字段权限** | 字段查看、编辑、脱敏权限 | 使用 Oinone 默认能力，无需扩展 |
| **数据权限** | 本人/团队/部门/全局数据权限 | **需要通过 Placeholder 扩展实现** |

### 1.2 核心概念

- **Placeholder 扩展**：通过继承 `AbstractPlaceHolderParser` 实现自定义占位符
- **权限规则配置**：通过 `DataPermissionRule` 模型配置权限范围
- **权限范围类型**：PERSONAL（个人）、TEAM（团队）、DEPARTMENT（部门）、GLOBAL（全局）、CUSTOM（自定义）
- **RSQL 协议**：使用 RSQL 语法生成查询条件，而非 SQL

### 1.3 RSQL 基础语法

RSQL 是 Oinone 使用的查询条件协议，常用操作符：

| 操作符 | 说明 | 示例 |
|--------|------|------|
| `==` | 等于 | `ownerId == 123` |
| `!=` | 不等于 | `status != 'DELETED'` |
| `=in=` | 包含于 | `organizationId =in= (1,2,3)` |
| `=out=` | 不包含于 | `status =out= ('DELETED','CANCELLED')` |
| `=lt=` | 小于 | `amount =lt= 1000` |
| `=le=` | 小于等于 | `amount =le= 1000` |
| `=gt=` | 大于 | `amount =gt= 100` |
| `=ge=` | 大于等于 | `amount =ge= 100` |
| `=like=` | 模糊匹配 | `name =like= 'test'` |
| `=isnull=` | 为空 | `deletedAt =isnull=` |
| `=notnull=` | 不为空 | `deletedAt =notnull=` |

逻辑运算符：
- `and` 或 `;`：逻辑与
- `or` 或 `,`：逻辑或

## 2. 数据权限模型定义

### 2.1 DataPermissionRule 模型

```java
@Model.model(DataPermissionRule.MODEL_MODEL)
@Model(displayName = "数据权限规则", labelFields = {"modelName"})
public class DataPermissionRule extends IdModel {

    private static final long serialVersionUID = 1L;
    public static final String MODEL_MODEL = "ocrm.DataPermissionRule";

    @Field.many2one
    @Field(displayName = "角色", summary = "关联角色", required = true)
    @Field.Relation(relationFields = {"roleId"}, referenceFields = {"id"})
    private AuthRole role;

    @Field(displayName = "角色ID", summary = "角色ID", invisible = true)
    private Long roleId;

    @Field.String
    @Field(displayName = "模型名称", summary = "模型名称", required = true)
    private String modelName;

    @Field.Enum
    @Field(displayName = "权限范围类型", summary = "权限范围类型", required = true)
    private ScopeTypeEnum scopeType;

    @Field.String
    @Field(displayName = "权限范围配置", summary = "权限范围配置(JSON)")
    private String scopeConfig;
}
```

### 2.2 ScopeTypeEnum 枚举

```java
@Base
@Dict(dictionary = ScopeTypeEnum.DICTIONARY, displayName = "权限范围类型", summary = "权限范围类型")
public enum ScopeTypeEnum implements IEnum<String> {

    PERSONAL("PERSONAL", "个人", "仅个人数据"),
    TEAM("TEAM", "团队", "团队及下属数据"),
    DEPARTMENT("DEPARTMENT", "部门", "部门及下属数据"),
    GLOBAL("GLOBAL", "全局", "所有数据"),
    CUSTOM("CUSTOM", "自定义", "RSQL 自定义范围");

    public static final String DICTIONARY = "ocrm.ScopeTypeEnum";

    private final String value;
    private final String displayName;
    private final String help;

    ScopeTypeEnum(String value, String displayName, String help) {
        this.value = value;
        this.displayName = displayName;
        this.help = help;
    }

    @Override
    public String value() {
        return value;
    }

    @Override
    public String displayName() {
        return displayName;
    }

    @Override
    public String help() {
        return help;
    }
}
```

## 3. Placeholder 扩展实现

### 3.1 基本结构

```java
@Component
public class OcrmDataPermissionPlaceholder extends AbstractPlaceHolderParser {

    private static final String DATA_PERMISSION_PLACEHOLDER = "${ocrmDataPermission}";
    private static final String NO_PERMISSION_CONDITION = "1==-1";
    private static final String ALL_DATA_CONDITION = "1==1";

    @Override
    protected String value() {
        try {
            Long userId = PamirsSession.getUserId();
            if (userId == null) {
                return NO_PERMISSION_CONDITION;
            }

            List<DataPermissionRule> rules = getDataPermissionRules(userId);
            if (rules == null || rules.isEmpty()) {
                return NO_PERMISSION_CONDITION;
            }

            return buildPermissionCondition(rules);
        } catch (Exception e) {
            return NO_PERMISSION_CONDITION;
        }
    }

    @Override
    public String namespace() {
        return DATA_PERMISSION_PLACEHOLDER;
    }

    @Override
    public Integer priority() {
        return -namespace().length();
    }

    @Override
    public Boolean active() {
        return true;
    }
}
```

**常量说明**：
- `NO_PERMISSION_CONDITION = "1==-1"`：无权限条件，返回空数据
- `ALL_DATA_CONDITION = "1==1"`：所有数据权限，返回全部数据

### 3.2 查询权限规则

```java
private List<DataPermissionRule> getDataPermissionRules(Long userId) {
    try {
        PamirsUser user = new PamirsUser().setId(userId).queryById();
        if (user == null || user.getRoles() == null || user.getRoles().isEmpty()) {
            return null;
        }

        List<AuthRole> roles = user.getRoles();
        List<Long> roleIds = new java.util.ArrayList<>();
        for (AuthRole role : roles) {
            if (role != null && role.getId() != null) {
                roleIds.add(role.getId());
            }
        }

        if (roleIds.isEmpty()) {
            return null;
        }

        LambdaQueryWrapper<DataPermissionRule> queryWrapper = Pops.<DataPermissionRule>lambdaQuery()
                .in(DataPermissionRule::getRoleId, roleIds);
        return new DataPermissionRule().queryList(queryWrapper);
    } catch (Exception e) {
        return null;
    }
}
```

### 3.3 构建权限条件

```java
private String buildPermissionCondition(List<DataPermissionRule> rules) {
    List<String> conditions = new java.util.ArrayList<>();
    
    for (DataPermissionRule rule : rules) {
        if (rule.getScopeType() == ScopeTypeEnum.GLOBAL) {
            return ALL_DATA_CONDITION;
        }
        
        if (rule.getScopeType() == ScopeTypeEnum.PERSONAL) {
            conditions.add(buildPersonalCondition());
        }
        
        if (rule.getScopeType() == ScopeTypeEnum.TEAM) {
            conditions.add(buildTeamCondition());
        }
        
        if (rule.getScopeType() == ScopeTypeEnum.DEPARTMENT) {
            conditions.add(buildDepartmentCondition());
        }
        
        if (rule.getScopeType() == ScopeTypeEnum.CUSTOM) {
            conditions.add(buildCustomCondition(rule.getScopeConfig()));
        }
    }

    String sharingCondition = buildSharingCondition();
    if (sharingCondition != null && !sharingCondition.isEmpty()) {
        conditions.add(sharingCondition);
    }

    if (conditions.isEmpty()) {
        return NO_PERMISSION_CONDITION;
    }

    if (conditions.size() == 1) {
        return conditions.get(0);
    }

    StringBuilder result = new StringBuilder();
    result.append("(").append(conditions.get(0));
    for (int i = 1; i < conditions.size(); i++) {
        result.append(") or (").append(conditions.get(i));
    }
    result.append(")");
    
    return result.toString();
}
```

**逻辑说明**：
- 所有权限规则之间使用 `or` 连接
- 如果有 GLOBAL 权限，直接返回 `1==1`
- 如果没有条件，返回 `1==-1`（无权限）
- 如果只有一个条件，直接返回该条件
- 如果有多个条件，用 `or` 连接所有条件
- 分享条件会自动添加到权限条件中

## 4. 权限范围实现规范

### 4.1 PERSONAL（个人权限）

**规则**：只能查看自己创建或负责的数据

```java
private String buildPersonalCondition() {
    Long userId = PamirsSession.getUserId();
    return "ownerId == " + userId;
}
```

**生成的 RSQL**：`ownerId == 123`

### 4.2 TEAM（团队权限）

**规则**：查看同一组织的数据

```java
private String buildTeamCondition() {
    Long userId = PamirsSession.getUserId();
    Organization organization = getUserOrganization(userId);
    if (organization != null) {
        return "organizationId == " + organization.getId();
    }
    return "ownerId == " + userId;
}

private Organization getUserOrganization(Long userId) {
    try {
        PamirsUser user = new PamirsUser().setId(userId).queryById();
        if (user == null) {
            return null;
        }

        LambdaQueryWrapper<Organization> queryWrapper = Pops.<Organization>lambdaQuery()
                .eq(Organization::getManagerId, userId)
                .last("LIMIT 1");
        List<Organization> organizations = new Organization().queryList(queryWrapper);
        if (organizations != null && !organizations.isEmpty()) {
            return organizations.get(0);
        }
    } catch (Exception e) {
    }
    return null;
}
```

**生成的 RSQL**：`organizationId == 1`

### 4.3 DEPARTMENT（部门权限）

**规则**：查看部门及子部门的数据（层级权限）

```java
private String buildDepartmentCondition() {
    Long userId = PamirsSession.getUserId();
    Organization organization = getUserManagedOrganization(userId);
    if (organization != null) {
        return buildOrganizationHierarchyCondition(organization);
    }
    return "ownerId == " + userId;
}

private Organization getUserManagedOrganization(Long userId) {
    try {
        PamirsUser user = new PamirsUser().setId(userId).queryById();
        if (user == null) {
            return null;
        }

        LambdaQueryWrapper<Organization> queryWrapper = Pops.<Organization>lambdaQuery()
                .eq(Organization::getManagerId, userId)
                .last("LIMIT 1");
        List<Organization> organizations = new Organization().queryList(queryWrapper);
        if (organizations != null && !organizations.isEmpty()) {
            return organizations.get(0);
        }
    } catch (Exception e) {
    }
    return null;
}

private String buildOrganizationHierarchyCondition(Organization organization) {
    try {
        String path = organization.getPath();
        if (path != null && !path.isEmpty()) {
            LambdaQueryWrapper<Organization> queryWrapper = Pops.<Organization>lambdaQuery()
                    .likeRight(Organization::getPath, path + "/");
            List<Organization> childOrganizations = new Organization().queryList(queryWrapper);
            
            if (childOrganizations != null && !childOrganizations.isEmpty()) {
                StringBuilder orgIds = new StringBuilder();
                orgIds.append("(").append(organization.getId());
                for (Organization child : childOrganizations) {
                    orgIds.append(",").append(child.getId());
                }
                orgIds.append(")");
                return "organizationId =in= " + orgIds.toString();
            }
        }

        return "organizationId == " + organization.getId();
    } catch (Exception e) {
        return NO_PERMISSION_CONDITION;
    }
}
```

**生成的 RSQL**：`organizationId =in= (1,2,3)`

**注意**：
- 使用 `managerId` 字段查询用户管理的组织，而不是 `createUid`
- 通过组织路径（path）实现层级权限，支持总部—区域—团队

### 4.4 GLOBAL（全局权限）

**规则**：查看所有数据

```java
private String buildGlobalCondition() {
    return ALL_DATA_CONDITION;
}
```

**生成的 RSQL**：`1==1`

### 4.5 CUSTOM（自定义权限）

**规则**：根据配置的 RSQL 生成自定义条件

```java
private String buildCustomCondition(String scopeConfig) {
    if (scopeConfig != null && !scopeConfig.isEmpty()) {
        return scopeConfig;
    }
    return NO_PERMISSION_CONDITION;
}
```

**生成的 RSQL**：从 `scopeConfig` 中读取，例如`organizationId =in= (1,2,3) and status != 'DELETED'`

### 4.6 SHARING（分享权限）

**规则**：根据 CustomerSharing 表中的记录生成分享条件

```java
private String buildSharingCondition() {
    try {
        Long userId = PamirsSession.getUserId();
        if (userId == null) {
            return null;
        }

        LambdaQueryWrapper<CustomerSharing> queryWrapper = Pops.<CustomerSharing>lambdaQuery()
                .eq(CustomerSharing::getUserId, userId);
        List<CustomerSharing> customerSharings = new CustomerSharing().queryList(queryWrapper);

        if (customerSharings == null || customerSharings.isEmpty()) {
            return null;
        }

        StringBuilder customerIds = new StringBuilder();
        customerIds.append("(");
        for (int i = 0; i < customerSharings.size(); i++) {
            if (i > 0) {
                customerIds.append(",");
            }
            customerIds.append(customerSharings.get(i).getCustomerId());
        }
        customerIds.append(")");

        return "id =in= " + customerIds.toString();
    } catch (Exception e) {
        return null;
    }
}
```

**生成的 RSQL**：`id =in= (1,2,3)`

**说明**：
- 从 `CustomerSharing` 表查询 `userId` 为当前用户 ID 的记录
- 获取所有记录中的 `customerId`
- 生成 RSQL 条件用于查询被分享的客户

## 5. 组织模型设计规范

### 5.1 必需字段

```java
@Model.model(Organization.MODEL_MODEL)
@Model(displayName = "组织", labelFields = {"name"})
public class Organization extends IdModel {

    @Field.String
    @Field(displayName = "组织名称", summary = "组织名称", required = true)
    private String name;

    @Field.String
    @Field(displayName = "组织编码", summary = "组织编码", required = true)
    private String code;

    @Field.many2one
    @Field(displayName = "上级组织", summary = "上级组织")
    @Field.Relation(relationFields = {"parentId"}, referenceFields = {"id"})
    private Organization parent;

    @Field(displayName = "上级组织ID", summary = "上级组织ID", invisible = true)
    private Long parentId;

    @Field.Integer
    @Field(displayName = "组织级别", summary = "组织级别")
    private Integer level;

    @Field.String
    @Field(displayName = "组织路径", summary = "组织路径")
    private String path;

    @Field.Enum
    @Field(displayName = "状态", summary = "组织状态", defaultValue = "ACTIVE")
    private OrganizationStatusEnum status;

    @Field.one2many
    @Field(displayName = "下级组织", summary = "下级组织")
    @Field.Relation(relationFields = {"id"}, referenceFields = {"parentId"})
    private List<Organization> children;

    @Field.many2one
    @Field(displayName = "组织管理者", summary = "组织负责人")
    @Field.Relation(relationFields = {"managerId"}, referenceFields = {"id"})
    private PamirsUser manager;

    @Field(displayName = "管理者ID", summary = "管理者ID", invisible = true)
    private Long managerId;
}
```

### 5.2 字段说明

| 字段 | 类型 | 说明 | 必需 |
|------|------|------|------|
| name | String | 组织名称 | ✅ |
| code | String | 组织编码 | ✅ |
| parent | many2one(Organization) | 上级组织 | ✅ |
| parentId | Long | 上级组织ID（invisible） | ✅ |
| level | Integer | 组织级别 | ✅ |
| path | String | 组织路径（用于层级权限） | ✅ |
| status | Enum | 组织状态 | ✅ |
| children | one2manyOne(Organization) | 下级组织 | ✅ |
| **manager** | **many2one(PamirsUser)** | **组织管理者** | ✅ **数据权限必需** |
| **managerId** | **Long** | **管理者ID（invisible）** | ✅ **数据权限必需** |

### 5.3 组织路径规范

组织路径用于实现层级权限，格式为：`/总部编码/区域编码/团队编码`

示例：
- 总部：`/HQ`
- 区域：`/HQ/REGION`
- 团队：`/HQ/REGION/TEAM`

## 6. 业务模型设计规范

### 6.1 必需字段

需要数据权限控制的业务模型必须包含以下字段：

```java
@Field.many2one
@Field(displayName = "所属组织", summary = "所属组织")
@Field.Relation(relationFields = {"organizationId"}, referenceFields = {"id"})
private Organization organization;

@Field(displayName = "组织ID", summary = "组织ID", invisible = true)
private Long organizationId;

@Field.many2one
@Field(displayName = "所属产品线", summary = "所属产品线")
@Field.Relation(relationFields = {"productLineId"}, referenceFields = {"id"})
private ProductLine productLine;

@Field(displayName = "产品线ID", summary = "产品线ID", invisible = true)
private Long productLineId;

@Field.many2one
@Field(displayName = "负责人", summary = "负责人")
@Field.Relation(relationFields = {"ownerId"}, referenceFields = {"id"})
private PamirsUser owner;

@Field(displayName = "负责人ID", summary = "负责人ID", invisible = true)
private Long ownerId;

@Field.many2many
@Field(displayName = "共享给", summary = "共享给其他用户")
@Field.Relation(relationFields = {"id"}, referenceFields = {"id"})
private List<PamirsUser> sharedUsers;
```

### 6.2 CustomerSharing 模型

客户共享模型用于记录客户分享给其他用户的信息：

```java
@Model.model(CustomerSharing.MODEL_MODEL)
@Model(displayName = "客户共享", labelFields = {"customer", "user"})
public class CustomerSharing extends IdModel {

    @Field.many2one
    @Field(displayName = "客户", summary = "共享的客户", required = true)
    @Field.Relation(relationFields = {"customerId"}, referenceFields = {"id"})
    private Customer customer;

    @Field(displayName = "客户ID", summary = "客户ID", invisible = true)
    private Long customerId;

    @Field.many2one
    @Field(displayName = "共享用户", summary = "被共享的用户", required = true)
    @Field.Relation(relationFields = {"userId"}, referenceFields = {"id"})
    private PamirsUser user;

    @Field(displayName = "用户ID", summary = "用户ID", invisible = true)
    private Long userId;

    @Field.Enum
    @Field(displayName = "权限级别", summary = "共享权限级别", defaultValue = "READ")
    private CustomerSharingPermissionEnum permissionLevel;

    @Field.Date
    @Field(displayName = "共享时间", summary = "共享时间")
    private Date shareDate;

    @Field.String
    @Field(displayName = "备注", summary = "共享备注")
    private String remark;
}
```

### 6.3 适用模型

需要数据权限控制的模型：
- Customer（客户）
- Lead（线索）
- Opportunity（机会）
- 其他业务模型

### 6.4 权限逻辑说明

数据权限基于以下维度：
1. **组织维度**：通过 `organizationId` 控制组织范围权限
2. **产品线维度**：通过 `productLineId` 控制产品线范围权限
3. **负责人维度**：通过 `ownerId` 控制个人/团队权限
4. **共享维度**：通过 `CustomerSharing` 模型控制数据共享权限

**共享权限逻辑**：
- 从 `CustomerSharing` 表查询 `userId` 为当前用户 ID 的记录
- 获取所有记录中的 `customerId`
- 生成 RSQL 条件：`id =in= (customerId1, customerId2, ...)`

## 7. 使用方式

### 7.1 RSQL 查询中使用

在 RSQL 查询中使用占位符：

```graphql
query {
  customerProxyQuery {
    queryPage(
      page: { currentPage: 1, pageSize: 10 }
      queryWrapper: {
        rsql: "${ocrmDataPermission}",
        queryData: {}
      }
    ) {
      content {
        id
        name
      }
    }
  }
}
```

系统会自动将 `${ocrmDataPermission}` 替换为实际的 RSQL 权限条件。

### 7.2 使用示例

```graphql
-- 个人权限：只能查看自己负责的客户
queryWrapper: { rsql: "${ocrmDataPermission}" }
-- 替换为：queryWrapper: { rsql: "ownerId == 123" }

-- 团队权限：查看同一组织的客户
queryWrapper: { rsql: "${ocrmDataPermission}" }
-- 替换为：queryWrapper: { rsql: "organizationId == 1" }

-- 部门权限：查看部门及子部门的客户（层级权限）
queryWrapper: { rsql: "${ocrmDataPermission}" }
-- 替换为：queryWrapper: { rsql: "organizationId =in= (1,2,3)" }

-- 全局权限：可以查看所有客户
queryWrapper: { rsql: "${ocrmDataPermission}" }
-- 替换为：queryWrapper: { rsql: "1==1" }

-- 自定义权限：根据配置的 RSQL 生成条件
queryWrapper: { rsql: "${ocrmDataPermission}" }
-- 替换为：queryWrapper: { rsql: "organizationId =in= (1,2,3) and status != 'DELETED'" }
```

### 7.3 复杂权限条件示例

结合多个权限维度：

```java
private String buildComplexCondition() {
    Long userId = PamirsSession.getUserId();
    Organization organization = getUserOrganization(userId);
    ProductLine productLine = getUserProductLine(userId);
    
    if (organization != null && productLine != null) {
        return "(organizationId == " + organization.getId() + 
               " and productLineId == " + productLine.getId() + 
               ") or ownerId == " + userId;
    }
    
    return "ownerId == " + userId;
}
```

**生成的 RSQL**：`(organizationId == 1 and productLineId == 2) or ownerId == 123`

## 8. 最佳实践

### 8.1 权限规则配置

1. **基于角色配置**：权限规则应该基于角色配置，而不是直接基于用户
2. **权限范围合理**：根据业务需求选择合适的权限范围类型
3. **自定义权限谨慎**：CUSTOM 类型需要管理员配置，确保 RSQL 语法正确

### 8.2 组织管理

1. **管理者字段**：使用 `managerId` 而不是 `createUid` 查询用户管理的组织
2. **层级权限**：通过组织路径（path）实现层级权限
3. **组织路径规范**：统一使用 `/总部/区域/团队` 格式

### 8.3 RSQL 语法规范

1. **使用标准操作符**：使用 RSQL 标准操作符，而非 SQL 操作符
2. **字符串引号**：字符串值使用单引号包裹
3. **逻辑运算符**：使用 `and`/`or` 而非 `;`/`,` 提高可读性
4. **括号优先级**：复杂条件使用括号明确优先级

### 8.4 性能优化

1. **缓存权限规则**：可以缓存用户的权限规则，避免重复查询
2. **索引优化**：为 `DataPermissionRule.roleId` 和 `Organization.managerId` 建立索引
3. **异常处理**：所有异常都应该返回 `EMPTY_CONDITION`，避免影响正常查询

### 8.5 安全考虑

1. **RSQL 语法校验**：CUSTOM 类型的 `scopeConfig` 应该经过 RSQL 语法校验
2. **权限最小化**：遵循最小权限原则，用户只能访问必要的数据
3. **审计日志**：记录权限规则的变更和使用情况

## 9. 验收标准

### 9.1 功能验收

- [ ] 所有权限范围类型都能正确生成 RSQL 条件
- [ ] 层级权限能正确匹配子组织
- [ ] 权限规则基于角色配置
- [ ] 异常情况下返回 `1==-1` 表示无权限

### 9.2 性能验收

- [ ] 权限条件生成响应时间 < 100ms
- [ ] 支持权限规则缓存
- [ ] 数据库查询优化

### 9.3 安全验收

- [ ] CUSTOM 类型 RSQL 条件经过语法校验
- [ ] 无 RSQL 注入风险
- [ ] 权限变更有审计日志

## 10. 参考资料

- [RSQL Service](/zh-cn/DevManual/Reference/Front-EndFramework/Services/RSQL-service.md)
- [网关协议 API](/zh-cn/DevManual/Reference/Back-EndFramework/AdvanceAPI/protocol-API.md)
- ORM-API-RSQL（Model、Field、Pops、Service）
- UX-API.md（Menu、View、Widget）
- model.md（模型设计）
