# 业务数据初始化规范

> **版本**: 1.1  
> **基准**: Oinone 框架模块数据初始化  
> **参考实现**: CRM标准版数据初始化

## 1. 概述

本规范定义了在 Oinone 框架中实现业务模块数据初始化的标准做法，确保：

1. 模块首次安装时自动初始化测试数据
2. 模块升级时默认不初始化，可通过配置启用
3. 支持重复初始化（幂等性）
4. 遵循 Oinone 框架最佳实践

## 2. 文件组织结构

```
{module}/
├── {module}-core/src/main/java/pro/shushi/oinone/{module}/core/
│   └── init/
│       ├── config/
│       │   └── {Module}DataInitProperties.java    # 配置属性类
│       ├── data/
│       │   ├── BaseDataInit.java                  # 基础初始化抽象类
│       │   ├── PamirsUserDataInit.java            # 用户数据初始化类（必须最先初始化）
│       │   ├── {Entity}DataInit.java              # 各实体数据初始化类
│       │   ├── {Entity}DataInit.java
│       │   └── ...
│       └── {Module}ModuleDataInit.java        # 主初始化类
```

### 2.1 命名规范

| 文件类型 | 命名规范 | 示例 |
|---------|---------|------|
| 配置属性类 | `{Module}DataInitProperties.java` | `OcrmDataInitProperties.java` |
| 基础初始化类 | `BaseDataInit.java` | `BaseDataInit.java` |
| 实体初始化类 | `{Entity}DataInit.java` | `ProductDataInit.java` |
| 主初始化类 | `{Module}ModuleDataInit.java` | `OcrmModuleDataInit.java` |

## 3. 配置属性类规范

### 3.1 类定义

```java
package pro.shushi.oinone.{module}.init.config;

import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.stereotype.Component;

@Component
@ConfigurationProperties(prefix = "{module}.data-init")
public class {Module}DataInitProperties {

    private boolean upgradeInitData = false;

    private boolean enable = true;

    public boolean isUpgradeInitData() {
        return upgradeInitData;
    }

    public void setUpgradeInitData(boolean upgradeInitData) {
        this.upgradeInitData = upgradeInitData;
    }

    public boolean isEnable() {
        return enable;
    }

    public void setEnable(boolean enable) {
        this.enable = enable;
    }
}
```

### 3.2 配置规范

在 `application-{profile}.yml` 中添加配置：

```yaml
{module}:
  data-init:
    upgrade-init-data: false
    enable: true
```

### 3.3 配置项说明

| 配置项 | 类型 | 默认值 | 说明 |
|--------|------|--------|------|
| `upgrade-init-data` | boolean | false | 升级时是否初始化数据（默认不初始化） |
| `enable` | boolean | true | 是否启用数据初始化功能 |

## 4. 基础初始化抽象类规范

### 4.1 类定义

```java
package pro.shushi.oinone.{module}.init.data;

import pro.shushi.pamirs.framework.connectors.data.sql.Pops;
import pro.shushi.pamirs.meta.annotation.fun.extern.Slf4j;
import pro.shushi.pamirs.meta.base.AbstractModel;

import java.util.List;
import java.util.function.Consumer;

@Slf4j
public abstract class BaseDataInit {

    protected void createOrUpdate(AbstractModel model) {
        model.createOrUpdate();
    }

    protected void createOrUpdateBatch(List<? extends AbstractModel> models) {
        if (models.isEmpty()) {
            return;
        }
        models.get(0).createOrUpdateBatch(models);
    }

    protected <T extends AbstractModel> void createOrUpdateWithQuery(
            T model,
            Consumer<T> queryConditionSetter,
            Consumer<T> updateFieldSetter
    ) {
        T existing = (T) model.getClass().newInstance();
        queryConditionSetter.accept(existing);
        List<T> list = existing.queryList();
        
        if (list.isEmpty()) {
            model.create();
        } else {
            T existingModel = list.get(0);
            updateFieldSetter.accept(existingModel);
            existingModel.update();
        }
    }

    public abstract void init();
}
```

### 4.2 方法说明

| 方法 | 说明 | 使用场景 |
|-----|------|---------|
| `createOrUpdate()` | 创建或更新单条数据，支持幂等性 | 当数据不存在时创建，存在时更新（依赖主键或唯一索引） |
| `createOrUpdateBatch()` | 批量创建或更新数据，支持幂等性 | 批量初始化数据（依赖主键或唯一索引） |
| `createOrUpdateWithQuery()` | 通过查询条件实现幂等性创建或更新 | 当模型没有主键或唯一索引时使用 |
| `init()` | 抽象方法，子类必须实现 | 定义具体的数据初始化逻辑 |

### 4.3 幂等性原理

#### 4.3.1 使用 createOrUpdate() 方法

- 如果模型有主键或唯一索引字段，且设置了这些字段的值，则：
  - 数据存在时执行更新操作
  - 数据不存在时执行新增操作
- 如果模型没有主键且没有唯一索引字段，或主键、唯一索引字段数据为空，则：
  - **每次都执行新增操作，会导致数据重复**

#### 4.3.2 使用 createOrUpdateWithQuery() 方法（推荐）

- 通过 LambdaQuery 查询条件判断数据是否存在
- 数据不存在时执行新增操作
- 数据存在时执行更新操作
- **适用于所有模型，无论是否有主键或唯一索引**

### 4.4 使用示例

#### 4.4.1 使用 createOrUpdate()（适用于有主键/唯一索引的模型）

```java
@Override
public void init() {
    List<Product> products = Arrays.asList(
        createProduct("PROD001", "企业版"),
        createProduct("PROD002", "标准版")
    );
    createOrUpdateBatch(products);
    log.info("初始化产品数据完成，共{}条", products.size());
}

private Product createProduct(String code, String name) {
    Product product = new Product();
    product.setCode(code);
    product.setName(name);
    return product;
}
```

#### 4.4.2 使用 createOrUpdateWithQuery()（适用于无主键/唯一索引的模型）

```java
@Override
public void init() {
    createFollowUp("电话跟进", "与客户进行电话沟通", new Date());
    createFollowUp("会议跟进", "与客户进行会议讨论", new Date());
    log.info("初始化跟进记录数据完成");
}

private void createFollowUp(String type, String content, Date date) {
    FollowUp followUp = new FollowUp();
    followUp.setType(type);
    followUp.setContent(content);
    followUp.setFollowUpDate(date);
    
    createOrUpdateWithQuery(
        followUp,
        query -> query.setType(type),
        existing -> {
            existing.setContent(content);
            existing.setFollowUpDate(date);
        }
    );
}
```

### 4.5 最佳实践建议

1. **优先使用 createOrUpdateWithQuery()**：即使模型有主键或唯一索引，也推荐使用此方法，因为更明确、更可控
2. **查询条件选择**：选择业务上唯一且稳定的字段作为查询条件（如 code、name 等）
3. **更新字段设置**：在 updateFieldSetter 中设置所有需要更新的字段
4. **批量处理**：对于批量初始化，可以循环调用 createOrUpdateWithQuery()

## 5. 实体数据初始化类规范

### 5.1 类定义模板

#### 5.1.1 适用于有主键/唯一索引的模型

```java
package pro.shushi.oinone.{module}.init.data;

import org.springframework.stereotype.Component;
import pro.shushi.oinone.{module}.api.model.{Entity};
import pro.shushi.oinone.{module}.api.enums.{Enum};
import pro.shushi.pamirs.framework.connectors.data.sql.Pops;
import pro.shushi.pamirs.meta.annotation.fun.extern.Slf4j;

import java.util.Arrays;
import java.util.List;

@Slf4j
@Component
public class {Entity}DataInit extends BaseDataInit {

    @Override
    public void init() {
        List<{Entity}> entities = Arrays.asList(
            create{Entity}(...),
            create{Entity}(...),
            ...
        );
        createOrUpdateBatch(entities);
        log.info("初始化{EntityDisplayName}数据完成，共{}条", entities.size());
    }

    private {Entity} create{Entity}(...) {
        {Entity} entity = new {Entity}();
        entity.setUniqueField(uniqueValue);
        entity.setField1(value1);
        entity.setField2(value2);
        return entity;
    }
}
```

#### 5.1.2 适用于无主键/唯一索引的模型（推荐）

```java
package pro.shushi.oinone.{module}.init.data;

import org.springframework.stereotype.Component;
import pro.shushi.oinone.{module}.api.model.{Entity};
import pro.shushi.oinone.{module}.api.enums.{Enum};
import pro.shushi.pamirs.meta.annotation.fun.extern.Slf4j;

import java.util.Date;

@Slf4j
@Component
public class {Entity}DataInit extends BaseDataInit {

    @Override
    public void init() {
        create{Entity}(...);
        create{Entity}(...);
        ...
        log.info("初始化{EntityDisplayName}数据完成");
    }

    private void create{Entity}(...) {
        {Entity} entity = new {Entity}();
        entity.setQueryField(queryValue);
        entity.setField1(value1);
        entity.setField2(value2);
        
        createOrUpdateWithQuery(
            entity,
            query -> query.setQueryField(queryValue),
            existing -> {
                existing.setField1(value1);
                existing.setField2(value2);
            }
        );
    }
}
```

### 5.2 命名规范

- 类名：`{Entity}DataInit`
- 日志前缀：`初始化{EntityDisplayName}数据完成`
- 方法名：`create{Entity}()`

### 5.3 数据创建规范

#### 5.3.1 使用 createOrUpdateBatch()（适用于有主键/唯一索引）

1. 使用 `Arrays.asList()` 创建数据列表
2. 每个数据通过 `create{Entity}()` 方法创建
3. **必须设置主键或唯一索引字段的值**
4. 使用 `createOrUpdateBatch()` 批量初始化
5. 初始化完成后记录日志

#### 5.3.2 使用 createOrUpdateWithQuery()（适用于无主键/唯一索引）

1. 每个数据通过单独的 `create{Entity}()` 方法创建
2. 在 `createOrUpdateWithQuery()` 中指定查询条件
3. 在 `updateFieldSetter` 中设置所有需要更新的字段
4. 初始化完成后记录日志

### 5.4 幂等性保证

| 模型类型 | 推荐方法 | 查询条件 | 更新字段 |
|---------|---------|---------|---------|
| 有主键/唯一索引 | `createOrUpdateBatch()` | 主键或唯一索引字段 | 所有字段 |
| 无主键/唯一索引 | `createOrUpdateWithQuery()` | 业务唯一字段（如 code、name） | 所有字段 |

**重要提示**：
- 使用 `createOrUpdateBatch()` 时，如果没有设置主键或唯一索引字段的值，**每次都会新增数据，导致重复**
- 使用 `createOrUpdateWithQuery()` 可以确保幂等性，**推荐使用**

### 5.5 关联数据查询规范

当初始化的数据需要关联其他已初始化的数据时：

```java
private Customer queryCustomerByName(String name) {
    List<Customer> customers = new Customer().queryList(
        Pops.<Customer>lambdaQuery().eq(Customer::getCompanyName, name)
    );
    return customers.isEmpty() ? null : customers.get(0);
}

private Opportunity queryOpportunityByName(String name) {
    List<Opportunity> opportunities = new Opportunity().queryList(
        Pops.<Opportunity>lambdaQuery().eq(Opportunity::getName, name)
    );
    return opportunities.isEmpty() ? null : opportunities.get(0);
}
```

**查询规范**：
- 使用 `Pops.<Model>lambdaQuery()` 构建查询条件
- 使用 `queryList()` 方法执行查询
- 处理空结果，返回 `null` 或空列表

## 5.6 PamirsUser 数据初始化规范

当业务实体需要关联用户（负责人、创建人、处理人等）时，必须先初始化 PamirsUser 数据。

### 5.6.1 PamirsUserDataInit 初始化类定义

```java
package pro.shushi.oinone.{module}.init.data;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pro.shushi.pamirs.user.api.enmu.UserSourceEnum;
import pro.shushi.pamirs.user.api.model.PamirsUser;
import pro.shushi.pamirs.user.api.service.UserService;
import pro.shushi.pamirs.framework.connectors.data.sql.Pops;
import pro.shushi.pamirs.meta.annotation.fun.extern.Slf4j;

import java.util.List;

@Slf4j
@Component
public class PamirsUserDataInit extends BaseDataInit {

    @Autowired
    private UserService userService;

    public static final String USER_ADMIN = "system_admin";
    public static final String USER_SALES_MANAGER = "sales_manager";
    public static final String USER_SALES_REP = "sales_rep";
    public static final String USER_SUPPORT = "support";

    @Override
    public void init() {
        createUser(USER_ADMIN, "系统管理员", "system.admin@oinone.com", "13800000001", "Admin@1234");
        createUser(USER_SALES_MANAGER, "销售经理", "sales.manager@oinone.com", "13800000002", "Sales@2024");
        createUser(USER_SALES_REP, "销售代表", "sales.rep@oinone.com", "13800000003", "Rep@2024");
        createUser(USER_SUPPORT, "客服专员", "support@oinone.com", "13800000004", "Support@2024");
        log.info("初始化用户数据完成，共4条");
    }

    private void createUser(String code, String name, String email, String phone, String password) {
        List<PamirsUser> existingUsers = new PamirsUser().queryList(
            Pops.<PamirsUser>lambdaQuery().eq(PamirsUser::getCode, code)
        );

        if (!existingUsers.isEmpty()) {
            log.info("用户已存在，跳过创建：{}", code);
            return;
        }

        PamirsUser user = new PamirsUser();
        user.setCode(code);
        user.setLogin(code);
        user.setName(name);
        user.setEmail(email);
        user.setPhone(phone);
        user.setActive(Boolean.TRUE);
        user.setSource(UserSourceEnum.MANUAL);
        user.setPassword(password);
        userService.create(user);
        log.info("创建用户成功：{} - {}", code, name);
    }
}
```

### 5.6.2 实现要点

| 要点 | 说明 | 推荐做法 |
|-----|------|---------|
| 幂等性 | 确保重复初始化不会创建重复用户的 | 使用 `Pops` 查询用户是否已存在，存在则跳过 |
| 密码设置 | 必须设置符合复杂度要求的密码 | 调用 `user.setPassword(password)` 设置密码，`userService.create()` 会自动验证密码复杂度 |
| 密码复杂度 | 8-32位，必须包含大写字母、小写字母和特殊字符 | 由 `userService.create()` 自动验证，不符合则抛出异常 |
| 创建方法 | 使用 `userService.create(user)` 而非 `user.create()` | `UserService.create()` 会保存到数据库并生成登录凭证 |
| 查询方式 | 使用 `Pops.lambdaQuery()` 构建查询条件 | 避免手写 SQL |
| 日志记录 | 记录创建成功和跳过的情况 | 便于调试和追踪 |

### 5.6.3 BaseDataInit 添加用户查询方法

在 `BaseDataInit` 类中添加用户查询方法：

```java
protected PamirsUser queryUserByCode(String code) {
    List<PamirsUser> users = new PamirsUser().queryList(
        Pops.<PamirsUser>lambdaQuery().eq(PamirsUser::getCode, code)
    );
    return users.isEmpty() ? null : users.get(0);
}
```

### 5.6.4 实体数据初始化中使用用户

在实体数据初始化类中，先查询用户，然后设置到实体中：

```java
@Override
public void init() {
    PamirsUser salesManager = queryUserByCode(PamirsUserDataInit.USER_SALES_MANAGER);
    PamirsUser salesRep = queryUserByCode(PamirsUserDataInit.USER_SALES_REP);

    List<Customer> customers = Arrays.asList(
        createCustomer("北京科技有限公司", ..., salesManager),
        createCustomer("上海智能科技有限公司", ..., salesRep)
    );
    createOrUpdateBatch(customers);
    log.info("初始化客户数据完成，共{}条", customers.size());
}

private Customer createCustomer(String name, ..., PamirsUser owner) {
    Customer customer = new Customer();
    customer.setCompanyName(name);
    customer.setOwner(owner);
    return customer;
}
```

### 5.6.5 常见用户关联字段命名

| 字段用途 | 常用字段名 | 说明 |
|---------|------------|------|
| 负责人 | `owner` | 客户、线索、机会、任务的负责人 |
| 创建人 | `createdBy` | 跟进记录、工单的创建人 |
| 处理人 | `handler` | 工单的处理人 |
| 参与人 | `participants` | 跟进记录的参与人（many2many） |
| 共享给 | `sharedUsers` | 客户共享给其他用户（many2many） |

### 5.6.6 初始化顺序要求

**PamirsUserDataInit 必须最先初始化**，因为其他实体可能依赖用户数据。

```java
private void initData() {
    pamirsUserDataInit.init();
    productDataInit.init();
    leadDataInit.init();
    customerDataInit.init();
    ...
}
```

### 5.6.7 密码安全要求

**密码复杂度规则**：
- 长度：8 至 32 位
- 必须包含：至少一位大写字母
- 必须包含：至少一位小写字母
- 必须包含：至少一位特殊字符

**注意**：密码复杂度由 `userService.create()` 自动验证，不符合要求时会抛出异常。

| 建议 | 说明 |
|-----|------|
| 测试环境 | 使用符合复杂度要求的测试密码，如 "Admin@1234"、"Sales@2024" 等 |
| 生产环境 | 建议使用强密码或通过配置文件注入 |
| 密码管理 | 避免在代码中硬编码密码，可考虑从配置读取 |
| 首次登录 | 用户首次登录后应提示修改默认密码 |

## 6. 主初始化类规范

### 6.1 类定义模板

```java
package pro.shushi.oinone.{module}.init;

import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pro.shushi.oinone.{module}.{Module}Module;
import pro.shushi.oinone.{module}.init.config.{Module}DataInitProperties;
import pro.shushi.oinone.{module}.init.data.*;
import pro.shushi.pamirs.boot.common.api.command.AppLifecycleCommand;
import pro.shushi.pamirs.boot.common.api.init.InstallDataInit;
import pro.shushi.pamirs.boot.common.api.init.UpgradeDataInit;
import pro.shushi.pamirs.boot.common.api.init.ReloadDataInit;
import pro.shushi.pamirs.meta.annotation.fun.extern.Slf4j;

import java.util.List;

@Slf4j
@Component
public class {Module}ModuleDataInit implements InstallDataInit, UpgradeDataInit, ReloadDataInit {

    @Autowired
    private {Module}DataInitProperties properties;

    @Autowired
    private {Entity1}DataInit entity1DataInit;

    @Autowired
    private {Entity2}DataInit entity2DataInit;

    @Autowired
    private ...

    @Override
    public boolean init(AppLifecycleCommand command, String version) {
        log.info("{Module}模块开始安装初始化数据，版本：{}", version);
        try {
            initData();
            log.info("{Module}模块安装初始化数据完成");
            return true;
        } catch (Exception e) {
            log.error("{Module}模块安装初始化数据失败", e);
            return false;
        }
    }

    @Override
    public boolean upgrade(AppLifecycleCommand command, String version, String existVersion) {
        log.info("{Module}模块开始升级初始化数据，当前版本：{}，已安装版本：{}", version, existVersion);
        
        if (!properties.isUpgradeInitData()) {
            log.info("{Module}模块升级时不初始化数据（配置：{module}.data-init.upgrade-init-data=false）");
            return true;
        }
        
        try {
            initData();
            log.info("{Module}模块升级初始化数据完成");
            return true;
        } catch (Exception e) {
            log.error("{Module}模块升级初始化数据失败", e);
            return false;
        }
    }

    @Override
    public boolean reload(AppLifecycleCommand command, String version) {
        log.info("{Module}模块开始重新加载初始化数据，版本：{}", version);
        try {
            initData();
            log.info("{Module}模块重新加载初始化数据完成");
            return true;
        } catch (Exception e) {
            log.error("{Module}模块重新加载初始化数据失败", e);
            return false;
        }
    }

    private void initData() {
        entity1DataInit.init();
        entity2DataInit.init();
        ...
    }

    @Override
    public List<String> modules() {
        return Lists.newArrayList({Module}Module.MODULE_MODULE);
    }

    @Override
    public int priority() {
        return 0;
    }
}
```

### 6.2 接口说明

| 接口 | 方法 | 说明 |
|-----|------|------|
| `InstallDataInit` | `init()` | 模块安装时调用，必须初始化数据 |
| `UpgradeDataInit` | `upgrade()` | 模块升级时调用，根据配置决定是否初始化 |
| `ReloadDataInit` | `reload()` | 模块重新加载时调用，必须初始化数据 |

### 6.3 方法说明

| 方法 | 说明 |
|-----|------|
| `init()` | 安装初始化入口，返回是否成功 |
| `upgrade()` | 升级初始化入口，返回是否成功 |
| `reload()` | 重新加载初始化入口，返回是否成功 |
| `initData()` | 内部方法，按顺序调用各实体初始化类 |
| `modules()` | 返回模块编码列表 |
| `priority()` | 返回执行优先级 |

## 7. 初始化顺序规范

### 7.1 依赖关系原则

1. **无依赖实体**：优先初始化（如产品、行业字典等）
2. **被依赖实体**：后初始化（如客户、机会等）
3. **引用关系**：many2one/one2many 关联的实体在引用方之后初始化

### 7.2 典型初始化顺序示例

```java
private void initData() {
    pamirsUserDataInit.init();
    productDataInit.init();
    leadDataInit.init();
    customerDataInit.init();
    contactDataInit.init();
    opportunityDataInit.init();
    taskDataInit.init();
    followUpDataInit.init();
    quotationDataInit.init();
    contractDataInit.init();
    ticketDataInit.init();
    campaignDataInit.init();
    targetDataInit.init();
}
```

### 7.3 初始化顺序决策树

```
┌── 用户（PamirsUser，必须最先初始化）
├── 产品（无依赖）
├── 线索（无依赖）
├── 客户（无依赖）
│   ├── 联系人（依赖客户）
│   ├── 机会（依赖客户）
│   │   └── 报价单（依赖机会、产品）
│   │       └── 合同（依赖报价单）
│   │           └── 回款计划（依赖合同）
│   ├── 任务（依赖客户、机会）
│   └── 跟进记录（依赖客户、机会、线索）
├── 工单（依赖客户、合同）
├── 营销活动（无依赖）
└── 销售目标/预测（无依赖）
```

## 8. Oinone 框架规范

### 8.1 导入规范

```java
import pro.shushi.pamirs.boot.common.api.init.Inst {DataInit,UpgradeDataInit,ReloadDataInit};
import pro.shushi.pamirs.boot.common.api.command.AppLifecycleCommand;
import pro.shushi.pamirs.meta.annotation.fun.extern.Slf4j;
import pro.shushi.pamirs.framework.connectors.data.sql.Pops;
```

### 8.2 日志规范

```java
@Slf4j

log.info("{Module}模块开始安装初始化数据，版本：{}", version);
log.info("初始化{EntityDisplayName}数据完成，共{}条", entities.size());
log.error("{Module}模块安装初始化数据失败", e);
```

日志级别说明：
- `log.info()`：正常流程日志
- `log.error()`：异常日志

### 8.3 查询规范

```java
new Model().queryList(Pops.<Model>lambdaQuery().eq(Model::getField, value));
```

禁止手写 SQL，必须使用：
- `queryList()`：列表查询
- `queryOne()`：单条查询
- `Pops.lambdaQuery()`：条件构建

### 8.4 异常处理规范

```java
@Override
public boolean init(AppLifecycleCommand command, String version) {
    log.info("{Module}模块开始安装初始化数据，版本：{}", version);
    try {
        initData();
        log.info("{Module}模块安装初始化数据完成");
        return true;
    } catch (Exception e) {
        log.error("{Module}模块安装初始化数据失败", e);
        return false;
    }
}
```

异常处理原则：
- 所有初始化方法必须 try-catch 包裹
- 捕获获常记录错误日志
- 返回 `false` 表示初始化失败
- 不要捕获 `RuntimeException`

### 8.5 组件注解规范

```java
@Component
@Autowired
@Slf4j
```

## 9. 数据设计规范

### 9.1 测试数据原则

1. **真实性**：数据尽量贴近真实业务场景
2. **完整性**：覆盖主要业务流程
3. **关联性**：数据之间建立正确的关联关系
4. **可读性**：数据名称和描述清晰易懂

### 9.2 数据量建议

| 实体类型 | 建议数据量 | 说明 |
|---------|---------|------|
| 字典/枚举类 | 5-10条 | 覆盖主要选项 |
| 主业务实体 | 5-10条 | 覆盖典型场景 |
| 关联实体 | 10-20条 | 根据主实体数量调整 |
| 明细实体 | 10-30条 | 如报价明细、回款计划 |

### 9.3 字段设置规范

```java
private {Entity} create{Entity}(...) {
    {Entity} entity = new {Entity}();
    entity.setField1(value1);
    entity.setField2(value2);
    entity.setEnumField(EnumValue);
    return entity;
}
```

字段设置原则：
- 使用 setter 方法设置字段
- 枚举值使用枚举常量
- 日期使用 `new Date()` 或计算日期
- 金额使用 `new BigDecimal("value")` 格式

## 10. 使用方式

### 10.1 首次安装

系统会自动调用 `init()` 方法初始化所有测试数据。

### 10.2 升级模块（默认不初始化）

由于配置 `upgrade-init-data: false`，升级时不会重新初始化数据。

### 10.3 升级模块（强制初始化）

在配置文件中设置：
```yaml
{module}:
  data-init:
    upgrade-init-data: true
```


## 11. 完整实现示例

### 11.1 CRM标准版实现结构

```
ocrm/ocrm-core/src/main/java/pro/shushi/oinone/ocrm/init/
├── config/
│   └── OcrmDataInitProperties.java
├── data/
│   ├── BaseDataInit.java
│   ├── PamirsUserDataInit.java
│   ├── ProductDataInit.java
│   ├── LeadDataInit.java
│   ├── CustomerDataInit.java
│   ├── ContactDataInit.java
│   ├── OpportunityDataInit.java
│   ├── TaskDataInit.java
│   ├── FollowUpDataInit.java
│   ├── QuotationDataInit.java
│   ├── ContractDataInit.java
│   ├── TicketDataInit.java
│   ├── CampaignDataInit.java
│   └── TargetDataInit.java
└── OcrmModuleDataInit.java
```

### 11.2 配置文件示例

```yaml
ocrm:
  data-init:
    upgrade-init-data: false
    enable: true
```

### 11.3 初始化数据统计

| 实体 | 数据量 | 说明 |
|-----|--------|------|
| PamirsUser | 4条 | 系统管理员、销售经理、销售代表、客服专员 |
| Product | 5条 | 企业版、标准版、实施服务、定制开发、年度维护 |
| Lead | 3条 | 官网咨询、展会咨询、转介绍 |
| Customer | 5条 | 不同行业、规模、等级、阶段 |
| Contact | 7条 | 每个客户1-2个联系人 |
| Opportunity | 3条 | 不同阶段（方案提交、商务谈判、赢单） |
| Task | 3条 | 回访、发送方案、跟进 |
| FollowUpUpRecord | 3条 | 电话、会议、拜访 |
| Quotation | 1条 | 包含2条明细 |
| Contract | 1条 | 包含2条回款计划 |
| Ticket | 2条 | 咨询、维修 |
| Campaign | 2条 | 邮件营销、短信营销 |
| SalesTarget | 2条 | 全公司目标、团队目标 |
| SalesForecast | 1条 | 全公司预测 |

## 12. 注意事项

### 12.1 编译注意事项

1. 确保所有导入的类和枚举都存在
2. 确保使用正确的 Oinone 注解
3. 确保字段类型与 Model 定义一致

### 12.2 运行注意事项

1. 首次安装前确保数据库已创建
2. 升级前确认配置是否正确
3. 查看日志确认初始化是否成功

### 12.3 调试注意事项

1. 可以通过修改配置启用升级初始化进行测试
2. 使用 `reload` 命令可以快速测试初始化逻辑
3. 检查数据库确认数据是否正确创建

### 12.4 幂等性测试

1. 多次执行安装或 reload 命令
2. 确认数据不会重复创建
3. 确认数据更新时正确覆盖

## 13. 最佳实践

1. **分模块初始化**：每个实体独立一个初始化类，便于维护
2. **优先使用 createOrUpdateWithQuery()**：确保幂等性，支持重复初始化，适用于所有模型
3. **谨慎使用 createOrUpdateBatch()**：仅适用于有主键或唯一索引的模型，且必须设置这些字段的值
4. **配置控制升级**：升级时默认不初始化，避免覆盖生产数据
5. **日志记录**：每个步骤都有清晰的日志记录
6. **异常处理**：所有初始化都有异常捕获和日志
7. **查询优化**：使用 Pops.lambdaQuery() 而非手写 SQL
8. **依赖管理**：按照依赖关系排序初始化
9. **数据真实性**：测试数据尽量贴近真实业务场景
10. **幂等性保证**：所有初始化逻辑都必须保证幂等性，多次执行不会产生重复数据

## 14. 参考文档

- [Oinone 模块数据初始化教程](file:///Users/mac/Documents/coding/6.0/trae/oinone-dev/oinone-backend-tutorials/.trae/source/oinone-docs/zh-cn/DevManual/Tutorials/init-module-data.md)
- [Oinone ORM-API 文档](file:///Users/mac/Documents/coding/6.0/trae/oinone-dev/oinone-backend-tutorials/.trae/source/oinone-docs/zh-cn/DevManual/Reference/Back-EndFramework/ORM-API.md)
- [Oinone 项目开发规范](file:///Users/mac/Documents/coding/6.0/trae/oinone-dev/oinone-backend-tutorials/.trae/rules/project_rules.md)
