# Oinone Action 开发规范

## 一、Action 方法参数四种情况

### 情况一：直接关联跳转
**场景**：页面选中 AModel，需要跳转到 BModel 的表单页面编辑后提交

**实现方式**：
- 重写 BModel 的 `construct` 方法
- 在 `construct` 方法中默认把 AModel 写到 BModel 对应的关联字段上
- 即为创建 BModel 时关联 AModel 的字段设置默认值

---

### 情况二：使用 TransientModel 传输（最常用）
**场景**：AModel 与 BModel 没有直接关联，或者不是为 BModel 设置默认值，且自定义方法入参有 AModel 和其他字段

**实现方式**：
1. 创建传输模型 `CModel extends TransientModel`
2. CModel 的字段为表单需要填写的内容
3. 在 CModel 中添加 AModel 的关联字段（使用 `@Field.many2one`）
4. 在 `AModelAction` 上添加 `@UxRouteButton` 注解，指向 `CModel`
5. 创建 `CModelAction` 类
6. 在 `CModelAction` 中实现 `construct(CModel data, AModel aModel)` 方法（**两个参数**）
7. 在 `CModelAction` 中实现业务方法（**只有一个参数**：`CModel data`），返回值也是 `CModel`

**关键点**：
- ✅ `@UxRouteButton` 放在 **`AModelAction`** 类上
- ✅ `value.model` 指向 `CModel.MODEL_MODEL`
- ✅ `action.name` 指向 `CModelAction` 中的方法名
- ✅ **TransientModel 使用 `@Field.many2one` 建立关联字段**，而不是单独维护 ID 字段
- ✅ `construct` 方法入参：**两个参数** `construct(CModel data, AModel aModel)`
- ✅ **业务方法入参：一个参数** `executeAction(CModel data)`
- ✅ **业务方法返回值：`CModel`**
- ✅ 在 `construct` 中设置关联字段 `data.setAModel(aModel)`
- ✅ 在业务方法中通过 `data.getAModel()` 获取源模型

---

### 情况三：单模型操作
**场景**：页面选中了 AModel，无需跳转到 BModel，直接进行操作，且操作所需入参都从 AModel 中获取

**基本规则**：
- ✅ **入参必须是 AModel**
- ✅ **返回值必须是 AModel**
- ✅ **业务逻辑处理需要的数据从 AModel 中获取**
- 返回修改后的 AModel

**识别特征**：
- 返回类型是简单类型（BigDecimal、Integer、Boolean、Map 等）而不是 AModel
- 入参包含多个参数（如 AModel、Date、PamirsUser 等）

#### 子场景 1：计算类操作
**场景**：评分、计算指标等操作，计算结果需要存储到模型字段

**实现方式**：
- ✅ 计算结果通过 setter 方法存储到模型字段中
- ✅ 如果计算结果适合存储到模型字段，直接存储，不需要额外消息提示
- ✅ 如果需要消息提示，可以在存储后添加消息

**示例**：
```java
@Action(displayName = "线索评分", label = "线索评分", bindingType = ViewTypeEnum.FORM)
public Lead scoreLead(Lead lead) {
    Integer score = leadService.scoreLead(lead);
    lead.setScore(score);
    return lead;
}

@Action(displayName = "计算达成率", label = "计算达成率", bindingType = ViewTypeEnum.TABLE)
public SalesTarget calculateAchievementRate(SalesTarget salesTarget) {
    BigDecimal rate = salesTargetService.calculateAchievementRate(salesTarget.getId());
    salesTarget.setAchievementRate(rate);
    return salesTarget;
}
```

#### 子场景 2：检查类操作
**场景**：检查重复、验证等操作，需要根据检查结果返回不同的消息

**实现方式**：
- ✅ 返回值必须是模型类型（如 `Lead`），不能是 `Boolean`
- ✅ 使用 `if-else` 根据检查结果发送不同级别的消息
- ✅ 检查不通过时使用 `ERROR` 级别
- ✅ 检查通过时使用 `SUCCESS` 级别
- ✅ 消息内容要清晰说明检查结果

**示例**：
```java
@Action(displayName = "检查重复线索", label = "检查重复线索", bindingType = ViewTypeEnum.FORM)
public Lead checkDuplicateLead(Lead lead) {
    Boolean isDuplicate = leadService.checkDuplicateLead(lead);
    if (isDuplicate) {
        PamirsSession.getMessageHub()
            .msg(Message.init()
                .setLevel(InformationLevelEnum.ERROR)
                .setMessage("检查不通过：发现重复线索"));
    } else {
        PamirsSession.getMessageHub()
            .msg(Message.init()
                .setLevel(InformationLevelEnum.SUCCESS)
                .setMessage("检查通过"));
    }
    return lead;
}
```

#### 子场景 3：预测类操作
**场景**：计算预测、统计等操作，需要向用户展示计算结果

**实现方式**：
- ✅ 返回值必须是模型类型（如 `Customer`），不能是 `Map` 或其他类型
- ✅ 使用消息形式展示计算结果，而不是修改模型字段
- ✅ 使用 `String.format()` 格式化消息内容
- ✅ 消息内容要包含所有关键的计算结果
- ✅ 如果计算结果不适合存储到模型字段，使用消息提示

**示例**：
```java
@Action(displayName = "预测收入", label = "预测收入", bindingType = ViewTypeEnum.TABLE)
public Customer predictRevenue(Customer customer) {
    Map<String, Object> revenueMap = opportunityService.predictRevenue(customer.getId());
    BigDecimal totalExpectedAmount = (BigDecimal) revenueMap.get("totalExpectedAmount");
    BigDecimal weightedAmount = (BigDecimal) revenueMap.get("weightedAmount");
    Integer opportunityCount = (Integer) revenueMap.get("opportunityCount");
    
    String message = String.format("预测收入结果：机会数量=%d，总预计金额=%s，加权金额=%s", 
        opportunityCount, totalExpectedAmount, weightedAmount);
    
    PamirsSession.getMessageHub()
        .msg(Message.init()
            .setLevel(InformationLevelEnum.SUCCESS)
            .setMessage(message));
    
    return customer;
}
```

#### 错误示例对比

**错误示例 1：返回简单类型**
```java
@Action(displayName = "线索评分", label = "评分", bindingType = ViewTypeEnum.FORM)
public Integer scoreLead(Lead lead) {
    return leadService.scoreLead(lead);
}
```

**错误示例 2：入参包含多个参数**
```从 salesTarget 中获取需要的数据
@Action(displayName = "计算个人预测", label = "计算", bindingType = ViewTypeEnum.TABLE)
public BigDecimal calculatePersonalForecast(PamirsUser user, Date startDate, Date endDate) {
    return salesForecastService.calculatePersonalForecast(user.getId(), startDate, endDate);
}
```

---

### 情况四：同模型批量操作
**场景**：入参两个但都是 AModel

**实现方式**：
- 返回值和入参都变成 `List<AModel>`

---

## 二、@UxRouteButton 使用规范

### 放置位置
- ✅ **正确**：放在 **`AModelAction`** 类上（源模型的 Action 类）
- ❌ **错误**：放在 `TransientModel` 类上、`TransientModelAction` 类上或源 Model 类上

### 注解结构
```java
@UxRouteButton(
    value = @UxRoute(
        model = XxxTransientModel.MODEL_MODEL,
        viewType = ViewTypeEnum.FORM,
        viewName = ViewConstants.Name.formView
    ),
    action = @UxAction(
        name = "executeAction",
        label = "操作名称",
        displayName = "操作名称",
        contextType = ActionContextTypeEnum.SINGLE,
        bindingType = ViewTypeEnum.TABLE,
        bindingView = ViewConstants.Name.tableView,
        invisible = ExpConstants.idValueNotExist
    )
)
```

### 字段说明
| 字段 | 说明 | 常用值 |
|------|------|----------|
| `value.model` | 目标模型（TransientModel） | `XxxTransientModel.MODEL_MODEL` |
| `value.viewType` | 视图类型 | `ViewTypeEnum.FORM` |
| `value.viewName` | 视图名称 | `ViewConstants.Name.formView` |
| `action.name` | 动作名称（对应 TransientModelAction 中的方法名） | `"executeAction"` |
| `action.label` | 按钮标签 | `"操作名称"` |
| `action.displayName` | 显示名称 | `"操作名称"` |
| `action.contextType` | 上下文类型 | `ActionContextTypeEnum.SINGLE` 或 `ActionContextTypeEnum.BATCH` |
| `action.bindingType` | 绑定视图类型 | `ViewTypeEnum.TABLE` |
| `action.bindingView` | 绑定视图名称 | `ViewConstants.Name.tableView` |
| `action.invisible` | 显隐表达式 | `ExpConstants.idValueNotExist`（选中时显示）或 `ExpConstants.idValueExist`（未选中时显示）|

---

## 三、TransientModelAction 方法签名规范

### construct 方法
**【重要】construct 方法必须使用 `@Function` 注解，不能使用 `@Action` 注解**

```java
@Function.Advanced(displayName = "初始化数据", type = FunctionTypeEnum.QUERY)
@Function(summary = "数据表单构造函数", openLevel = {FunctionOpenEnum.LOCAL, FunctionOpenEnum.API, FunctionOpenEnum.REMOTE})
public XxxTransientModel construct(XxxTransientModel data, AModel aModel) {
    // 两个参数：data 是表单数据，aModel 是源模型
    // 设置关联字段，跳转时，many2one字段，前端只传 ID，其他字段在服务端获取
    AModel fullAModel = new AModel().setId(aModel.getId()).queryById();
    data.setAModel(fullAModel);
    return data;
}
```

### 业务方法
```java
@Action(displayName = "执行操作", label = "执行", bindingType = ViewTypeEnum.FORM)
public XxxTransientModel executeAction(XxxTransientModel data) {
    // 只有一个参数：data
    // 返回值也是 XxxTransientModel
    // 通过关联字段获取源模型
    xxxService.executeAction(aModel.getAModelId(), data.getField1(), data.getField2());
    return data;
}
```
```java
@Action(displayName = "执行操作", label = "执行", bindingType = ViewTypeEnum.FORM)
public XxxTransientModel executeAction(XxxTransientModel data) {
    // 只有一个参数：data
    // 返回值也是 XxxTransientModel
    // 通过关联字段获取源模型
    AModel aModel = new AModel();
    // 表单提交时，many2one字段，前端只传它的relationFields
    aModel.setId(data.getAModelId());
    xxxService.executeAction(aModel, data.getField1(), data.getField2());
    return data;
}
```

---

## 四、完整示例（情况二）

### 1. TransientModel
```java
@Base
@Model
@Model.Advanced(name = "leadConvertTransientModel")
@Model.model(LeadConvertTransientModel.MODEL_MODEL)
public class LeadConvertTransientModel extends TransientModel {
    
    public static final String MODEL_MODEL = "ocrm.LeadConvertTransientModel";
    
    // 使用关联字段，而不是单独的 ID 字段
    @Field.many2one
    @Field(displayName = "线索", summary = "线索")
    @Field.Relation(relationFields = {"leadId"}, referenceFields = {"id"})
    private Lead lead;

    @Field(displayName = "线索ID", summary = "线索ID", invisible = true)
    private Long leadId;
    
    @Field.String
    @Field(displayName = "客户名称", summary = "客户名称", required = true)
    private String customerName;
    
    @Field.String
    @Field(displayName = "联系人姓名", summary = "联系人姓名", required = true)
    private String contactName;
    
    @Field.String
    @Field(displayName = "机会名称", summary = "机会名称", required = true)
    private String opportunityName;
}
```

### 2. AModelAction（添加 @UxRouteButton）
```java
@Component
@Model.model(Lead.MODEL_MODEL)
@UxRouteButton(
    value = @UxRoute(
        model = LeadConvertTransientModel.MODEL_MODEL,
        viewType = ViewTypeEnum.FORM,
        viewName = ViewConstants.Name.formView
    ),
    action = @UxAction(
        name = "convertToCustomer",
        label = "转化为客户",
        displayName = "转化为客户",
        contextType = ActionContextTypeEnum.SINGLE,
        bindingType = ViewTypeEnum.TABLE,
        bindingView = ViewConstants.Name.tableView,
        invisible = ExpConstants.idValueNotExist
    )
)
public class LeadAction {
    
    @Autowired
    private LeadService leadService;
    
    // 其他方法...
}
```

### 3. TransientModelAction
```java
@Component
@Model.model(LeadConvertTransientModel.MODEL_MODEL)
public class LeadConvertTransientModelAction {
    
    @Autowired
    private LeadService leadService;
    
    // construct 方法：两个参数
    @Function.Advanced(displayName = "初始化数据", type = FunctionTypeEnum.QUERY)
    @Function(summary = "数据表单构造函数", openLevel = {FunctionOpenEnum.LOCAL, FunctionOpenEnum.API, FunctionOpenEnum.REMOTE})
    public LeadConvertTransientModel construct(LeadConvertTransientModel data, Lead lead) {
        if (lead != null && lead.getId() != null) {
            // 设置关联字段
            Lead fullLead = new Lead().setId(lead.getId()).queryById();
            // 跳转时，many2one字段，前端只传 ID，其他字段在服务端获取
            data.setLead(fullLead);
            if (fullLead != null) {
                if (data.getCustomerName() == null && fullLead.getCompanyName() != null) {
                    data.setCustomerName(fullLead.getCompanyName());
                }
                if (data.getContactName() == null && fullLead.getContactName() != null) {
                    data.setContactName(fullLead.getContactName());
                }
                if (data.getOpportunityName() == null && fullLead.getName() != null) {
                    data.setOpportunityName(fullLead.getName());
                }
            }
        }
        return data;
    }
    
    // 业务方法：只有一个参数，返回值也是 LeadConvertTransientModel
    @Action(displayName = "转化为客户", label = "转化", bindingType = ViewTypeEnum.FORM)
    public LeadConvertTransientModel convertToCustomer(LeadConvertTransientModel data) {
        Lead lead = new Lead();
        // 表单提交时，many2one字段，前端只传它的relationFields
        lead.setId(data.getLeadId());
        leadService.convertToCustomer(lead, data.getCustomerName(), data.getContactName(), data.getOpportunityName());
        return data;
    }
}
```

---

## 五、常见错误

| 错误 | 正确 |
|------|------|
| `@UxRouteButton` 放在 TransientModelAction 类上 | 放在 **`AModelAction`** 类上 |
| `@UxRouteButton` 放在 Model 类上 | 放在 **`AModelAction`** 类上 |
| 业务方法有两个参数（data, aModel） | **业务方法只有一个参数：`data`** |
| 业务方法返回 AModel | **业务方法返回 `TransientModel`** |
| `construct` 方法只有一个参数 | `construct` 方法有两个参数：`(data, aModel)` |
| `construct` 方法使用 `@Action` 注解 | **`construct` 方法必须使用 `@Function` 注解** |
| 使用 `Menu` 方式实现页面跳转 | 使用 `@UxRouteButton` 注解在 `AModelAction` 上 |
| 批量操作使用单参数 | 使用 `List<Model>` 作为参数 |
| `invisible` 表达式错误 | `ExpConstants.idValueNotExist`（选中时显示）|
| TransientModel 使用单独的 ID 字段 | **使用 `@Field.many2one` 建立关联字段** |

---

## 六、@@Action.Advanced invisible 设置规范

### 基本规则
当 Action 在 **FORM 表单** 中操作**已存在的模型数据**时，必须设置 `invisible = ExpConstants.idValueNotExist`。

### 何时需要设置 invisible = ExpConstants.idValueNotExist
- ✅ **需要设置**：操作已存在的模型数据（如：分配、接受、关闭、审批、签署等）
- ❌ **不需要设置**：创建新数据的操作（如：创建工单、创建产品等）

### 判断方法
查看 Action 方法实现：
- 如果方法中调用 `model.getId()` 获取 ID，说明操作的是已存在的数据 → **需要设置**
- 如果方法中直接使用传入的 model 对象创建新数据 → **不需要设置**

### 示例

#### 需要设置 invisible = ExpConstants.idValueNotExist
```java
@Action(displayName = "分配工单", label = "分配", bindingType = ViewTypeEnum.FORM)
public Ticket assignTicket(Ticket ticket) {
    return ticketService.assignTicket(ticket.getId(), ticket.getHandler());
}
```

#### 不需要设置（创建操作）
```java
@Action.Advanced(name = FunctionConstants.create, type = FunctionTypeEnum.CREATE, managed = true, invisible = ExpConstants.idValueExist, check = true)
@Action(displayName = "创建工单", label = "创建", bindingType = ViewTypeEnum.FORM)
@Function(name = FunctionConstants.create)
@Function.fun(FunctionConstants.create)
public Ticket createTicket(Ticket ticket) {
    return ticketService.createTicket(ticket);
}
```

### 特殊情况
- **检查类操作**（如：检查重复线索、检查重复客户）：这类操作不修改数据，可以根据业务需求决定是否设置 invisible

---

## 七、Action label 字段规范

### 基本规则
**Action 的 label 字段是给用户看的，必须见名知意，不能是简单和笼统的描述。**

### 禁止使用的 label
- ❌ `"创建"` - 不明确创建什么对象
- ❌ `"检查"` - 不明确检查什么
- ❌ `"计算"` - 不明确计算什么
- ❌ `"标记"` - 不明确标记为什么
- ❌ `"更新"` - 不明确更新什么
- ❌ `"删除"` - 不明确删除什么（除非是标准的 CRUD 操作）

### 正确的 label 示例
| 错误的 label | 正确的 label | 说明 |
|-------------|-------------|------|
| `"创建"` | `"创建报价"`、`"创建工单"`、`"创建产品"` | 明确创建的对象类型 |
| `"检查"` | `"检查重复客户"`、`"检查重复线索"` | 明确检查的内容 |
| `"计算"` | `"计算达成率"`、`"计算ROI"` | 明确计算的指标 |
| `"标记"` | `"标记为主联系人"` | 明确标记的目标 |
| `"更新"` | `"更新客户阶段"` | 明确更新的字段或属性 |

### 特殊情况
- **标准的 CRUD 操作**（如：创建、更新、删除）：如果 `displayName` 已经明确说明操作对象，label 可以使用简短形式，但建议保持一致
- **操作类动词**（如：分配、接受、关闭、审批、签署）：这些动词本身已经明确操作类型，可以使用简短 label

### 示例对比

#### 错误示例
```java
@Action(displayName = "计算达成率", label = "计算", bindingType = ViewTypeEnum.TABLE)
public SalesTarget calculateAchievementRate(SalesTarget salesTarget) {
    // ...
}

@Action(displayName = "创建报价", label = "创建", bindingType = ViewTypeEnum.FORM)
public Quotation createQuotation(Quotation quotation) {
    // ...
}
```

#### 正确示例
```java
@Action(displayName = "计算达成率", label = "计算达成率", bindingType = ViewTypeEnum.TABLE)
public SalesTarget calculateAchievementRate(SalesTarget salesTarget) {
    // ...
}

@Action(displayName = "创建报价", label = "创建报价", bindingType = ViewTypeEnum.FORM)
public Quotation createQuotation(Quotation quotation) {
    // ...
}
```

---

## 八、消息提示规范

### 基本规则
**Action 方法中需要向用户反馈操作结果时，使用 `PamirsSession.getMessageHub()` 发送消息。**

### 涉及类
```java
import pro.shushi.pamirs.meta.api.session.PamirsSession;
import pro.shushi.pamirs.meta.api.dto.common.Message;
import pro.shushi.pamirs.meta.enmu.InformationLevelEnum;
```

### 消息级别
| 级别 | 说明 | 使用场景 |
|--------|------|----------|
| `InformationLevelEnum.SUCCESS` | 成功消息 | 操作成功、检查通过等 |
| `InformationLevelEnum.ERROR` | 错误消息 | 操作失败、检查不通过等 |
| `InformationLevelEnum.WARNING` | 警告消息 | 需要注意的情况 |
| `InformationLevelEnum.INFO` | 信息消息 | 一般提示信息 |

### 使用方式
```java
PamirsSession.getMessageHub()
    .msg(Message.init()
        .setLevel(InformationLevelEnum.SUCCESS)
        .setMessage("操作成功"));
```

### 注意事项
- ✅ 消息提示主要用于检查类操作、预测类操作等需要向用户展示结果的场景
- ✅ 计算类操作如果结果适合存储到模型字段，直接存储，不需要额外消息提示
- ✅ 具体的实现方式请参考"情况三：单模型操作"中的各子场景

---

## 九、核心要点总结

1. ✅ `@UxRouteButton` 放在 **`AModelAction`** 类上
2. ✅ **TransientModel 使用 `@Field.many2one` 建立关联字段**，而不是单独维护 ID 字段
3. ✅ `construct` 方法有**两个参数**：`(TransientModel data, AModel aModel)`
4. ✅ **`construct` 方法必须使用 `@Function` 注解，不能使用 `@Action` 注解**
5. ✅ 在 `construct` 中设置关联字段 `data.setAModel(aModel)`
6. ✅ 业务方法有**一个参数**：`(TransientModel data)`
7. ✅ 业务方法返回值是 **`TransientModel`**
8. ✅ 在业务方法中通过 `data.getAModel()` 获取源模型
9. ✅ **FORM 表单中操作已存在的模型数据时，必须设置 `invisible = ExpConstants.idValueNotExist`**
10. ✅ **Action 的 label 字段必须见名知意，不能使用简单和笼统的描述（如："创建"、"检查"、"计算"等）**
11. ✅ **单模型操作的 Action 方法，返回值必须是模型类型，不能是简单类型（Integer、Boolean、BigDecimal、Map 等）**
12. ✅ **计算结果适合存储到模型字段时，通过 setter 方法存储；不适合时，使用 `PamirsSession.getMessageHub()` 发送消息**
13. ✅ **检查类操作使用 `if-else` 根据结果发送不同级别的消息（ERROR/SUCCESS）**
