# Oinone CRM 开发规范

> **版本**: 2.0 | **基准**: Oinone 官方文档 | **强制**: 参考 `.trae/source/oinone-docs/` 官文文档

## 1. Service 接口规范

**模板**：
```java
@Fun(ServiceName.FUN_NAMESPACE)
public interface ServiceName {
    String FUN_NAMESPACE = "模块.ServiceName";
    
    @Function
    返回类型 方法名(参数列表);
}
```

**规范要点**：
- 使用 `@Fun` 注解标记接口
- 定义 `FUN_NAMESPACE` 常量（格式：`模块.ServiceName`）
- 使用 `@Function` 注解标记所有公共方法
- 避免使用 get/set/unset 开头的方法名
- 使用动词+名词的命名方式（如：convertToCustomer、assignLead、scoreLead）
- 非必要不重复数据管理器的功能（create、update、delete 由 Oinone 默认提供）

## 2. Service 实现类规范

**模板**：
```java
@Service
@Fun(ServiceName.FUN_NAMESPACE)
@PamirsTransactional
public class ServiceNameImpl implements ServiceName {
    
    @Autowired
    private OtherService otherService;
    
    @Function
    @Override
    public 返回类型 方法名(参数列表) {
    }
}
```

**规范要点**：
- 使用 `@Service` 注解标记为 Spring 服务
- 使用 `@Fun` 注解指定服务命名空间
- 使用 `@PamirsTransactional` 注解标记事务管理
- 使用 `@Function` 注解标记所有公共方法
- 使用 `@Override` 注解标记接口实现
- 参数校验（检查 null 和空值）
- 权限检查（检查业务规则）
- 使用 `RuntimeException` 或 `PamirsException` 抛出异常

## 3. Action 类规范

**基本模板**：
```java
@Component
@Model.model(Model.MODEL_MODEL)
public class ModelAction {
    
    @Autowired
    private ServiceName serviceName;
    
    @Action(displayName = "展示名称", label = "名称", bindingType = {ViewTypeEnum.TABLE})
    public 返回类型 方法名(Model model) {
        return serviceName.方法名(从模型对象提取的参数);//单对象，选中对象直接调用模型方法
    }

    @Action(displayName = "展示名称", label = "名称", bindingType = {ViewTypeEnum.FORM})
    public 返回类型 方法名(Model model) {
        return serviceName.方法名(从模型对象提取的参数);//单对象，编辑以后再调用模型方法
    }

    @Action(displayName = "展示名称", label = "名称", bindingType = ViewTypeEnum.TABLE, contextType = ActionContextTypeEnum.SINGLE_AND_BATCH)
    public 返回类型 方法名(List<Model> models) {
        return serviceName.方法名(从模型对象提取的参数);//多对象，选中对象直接调用模型方法
    }
}
```

**规范要点**：
- 使用 `@Component` 注解标记为 Spring 组件
- 使用 `@Model.model(Model.MODEL_MODEL)` 注解关联到对应的 Model
- 使用 `@Action` 注解标记为 Action
- 使用 `@Autowired` 注解注入 Service 依赖
- Action 方法参数使用模型对象（符合 GraphQL 协议）
- 只承担协调者角色，不直接用 Pops 或操作事务
- 调用 Service 方法执行业务逻辑

### 3.2 Action 配置规则

**重要规则**：
- ✅ 正常情况下不要同时配置 `@Action` 和 `@Function/@Function.fun`
- ✅ 除非是为了重写模型的新增/修改/删除行为
- ✅ 参考DefaultWriteWithFieldApi的写法

**重写模型CRUD行为的配置**（create/update/delete）：

```java
@Action.Advanced(name = FunctionConstants.create, type = FunctionTypeEnum.CREATE, managed = true, invisible = ExpConstants.idValueExist, check = true)
@Action(displayName = "创建产品", label = "创建", bindingType = ViewTypeEnum.FORM)
@Function(name = FunctionConstants.create)
@Function.fun(FunctionConstants.create)
public Product createProduct(Product product) {
    return productService.createProduct(product);
}

@Action.Advanced(name = FunctionConstants.update, type = FunctionTypeEnum.UPDATE, managed = true, invisible = ExpConstants.idValueNotExist, check = true)
@Action(displayName = "更新产品", label = "更新", bindingType = ViewTypeEnum.FORM)
@Function(name = FunctionConstants.update)
@Function.fun(FunctionConstants.update)
public Product updateProduct(Product product) {
    return productService.updateProduct(product);
}

@Action.Advanced(name = FunctionConstants.delete, type = FunctionTypeEnum.DELETE, managed = true, check = true)
@Action(displayName = "删除产品", label = "删除", bindingType = ViewTypeEnum.TABLE, contextType = ActionContextTypeEnum.SINGLE_AND_BATCH)
@Function(name = FunctionConstants.delete)
@Function.fun(FunctionConstants.deleteWithFieldBatch)
public List<Product> deleteProductBatch(List<Product> productList) {
    return productService.deleteProductBatch(productList);
}
```

**自定义业务操作**（不重写CRUD）：
```java
@Action(displayName = "分配工单", label = "分配", bindingType = ViewTypeEnum.FORM)
public Ticket assignTicket(Ticket ticket) {
    return ticketService.assignTicket(ticket.getId(), ticket.getHandler());
}
```

**重要规则**：
- 重写CRUD时，同时配置 `@Action.Advanced`、`@Action`、`@Function`、`@Function.fun`
- 自定义业务操作时，只配置 `@Action`（不需要 `@Function` 和 `@Function.fun`）

### 3.2 bindingType 配置

- `ViewTypeEnum.TABLE` - 表格操作（如删除、批量操作）
- `ViewTypeEnum.FORM` - 表单操作（如创建、更新、分配等）
- 可以同时配置 TABLE 和 FORM

### 3.3 contextType 配置

- `ActionContextTypeEnum.SINGLE` - 单行操作（默认）
- `ActionContextTypeEnum.BATCH` - 多行操作（只在多行出现）
- `ActionContextTypeEnum.SINGLE_AND_BATCH` - 单行和多行（支持单条和批量操作）
- 如果入参是 `List<Model>`，配置成 `SINGLE_AND_BATCH`

### 3.4 批量删除特殊配置

```java
@Action.Advanced(name = FunctionConstants.delete, type = FunctionTypeEnum.DELETE, managed = true, check = true)
@Action(displayName = "删除产品", label = "删除", bindingType = ViewTypeEnum.TABLE, contextType = ActionContextTypeEnum.SINGLE_AND_BATCH)
@Function(name = FunctionConstants.delete)
@Function.fun(FunctionConstants.deleteWithFieldBatch)
public List<Product> deleteProductBatch(List<Product> productList) {
    return productService.deleteProductBatch(productList);
}
```

**规范要点**：
- 使用 `FunctionConstants.deleteWithFieldBatch` 而不是 `FunctionConstants.delete`
- 入参类型为 `List<Product>`
- 配置 `contextType = ActionContextTypeEnum.SINGLE_AND_BATCH`

## 4. Action 参数规范（GraphQL 协议）

**重要**：Action 的入参必须是模型对象，因为与前端的交互是基于 GraphQL 协议。

**规范要点**：
- Action 方法参数使用模型对象（如：`Lead lead`、`Customer customer`）
- 在 Action 中从模型对象提取需要的参数（如：`lead.getId()`、`lead.getOwner()`）
- 将提取的参数传递给 Service 方法
- 避免使用原始类型参数（如：`Long leadId`）

**正确示例**：
```java
@Action
public Lead assignLead(Lead lead) {
    return leadService.assignLead(lead.getId(), lead.getOwner());
}
```

**错误示例**：
```java
@Action
public Lead assignLead(Long leadId, PamirsUser owner) {
    return leadService.assignLead(leadId, owner);
}
```

## 5. 方法命名规范

**推荐模式**（动词+名词）：
- convertToCustomer、assignLead、scoreLead、mergeCustomer
- shareCustomer、advanceStage、loseOpportunity、completeTask
- createAutoTask、markAsPrimaryContact

**禁止模式**：
- getxxx - 用于查询，应该由数据管理器提供
- setxxx - 用于设置属性，不符合业务动作语义
- unsetxxx - 用于取消设置，不符合业务动作语义

## 6. 数据操作规范

**规范要点**：
- 使用 `updateById()` 而不是 `update()`
- 使用 `new Model().setId(id).queryById()` 查询
- 使用 `Pops.<Model>lambdaQuery()` 进行复杂查询
- **【强制】复杂查询只支持普通字段，不支持关联对象字段**

**示例**：
```java
Customer customer = new Customer();
customer.setCompanyName(customerName);
customer.create();

Lead lead = new Lead().setId(leadId).queryById();
lead.setOwner(owner);
lead.updateById();

LambdaQueryWrapper<Lead> queryWrapper = Pops.<Lead>lambdaQuery()
        .eq(Lead::getPhone, phone)
        .ne(Lead::getId, lead.getId());
List<Lead> leads = new Lead().queryList(queryWrapper);
```

### 6.1 复杂查询字段规范

**重要规则**：
- ✅ 使用 `::getCustomerId`（普通字段）
- ❌ 使用 `::getCustomer`（关联对象字段，不支持）

**原因**：Oinone 复杂查询只支持普通字段，不支持 many2one/one2many/many2many 等关联对象字段

**正确示例**：
```java
LambdaQueryWrapper<Opportunity> queryWrapper = Pops.<Opportunity>lambdaQuery()
        .eq(Opportunity::getCustomerId, customerId);
List<Opportunity> opportunities = new Opportunity().queryList(queryWrapper);
```

**错误示例**：
```java
LambdaQueryWrapper<Opportunity> queryWrapper = Pops.<Opportunity>lambdaQuery()
        .eq(Opportunity::getCustomer, customerId);
List<Opportunity> opportunities = new Opportunity().queryList(queryWrapper);
```

## 7. Java 枚举比较规范

**规范要点**：
- 使用 `.equals()` 方法而不是 `==` 操作符
- 避免空指针异常（先检查对象是否为 null）

**示例**：
```java
TaskStatusEnum.DONE.equals(task.getStatus())  // ✅ 正确
task.getStatus() == TaskStatusEnum.DONE  // ❌ 错误
```

## 8. 快速检查清单

### Service 检查项
- [ ] Service 接口使用 `@Fun` 注解
- [ ] Service 接口定义 `FUN_NAMESPACE` 常量
- [ ] Service 接口使用 `@Function` 注解
- [ ] Service 实现类使用 `@Service`、`@Fun`、`@PamirsTransactional` 注解
- [ ] Service 实现类使用 `@Function`、`@Override` 注解
- [ ] 数据操作使用 `updateById()` 而不是 `update()`
- [ ] 枚举比较使用 `.equals()` 方法
- [ ] 避免重复数据管理器的功能（create、update、delete）
- [ ] 复杂查询使用普通字段（如 ::getCustomerId），不使用关联对象字段（如 ::getCustomer）

### Action 检查项
- [ ] Action 类使用 `@Component`、`@Model.model()`、`@Action` 注解
- [ ] Action 类使用方法名以动词开头（不是 get/set/unset）
- [ ] Action 参数使用模型对象（符合 GraphQL 协议）
- [ ] 如果是重写模型CRUD行为，同时配置 `@Action.Advanced`、`@Action`、`@Function`、`@Function.fun`
- [ ] 如果是自定义业务操作，只配置 `@Action`（不需要 `@Function` 和 `@Function.fun`）
- [ ] `@Action` 配置了 `displayName`、`label`、`bindingType`
- [ ] 批量操作配置了 `contextType = ActionContextTypeEnum.SINGLE_AND_BATCH`
- [ ] 批量删除使用 `FunctionConstants.deleteWithFieldBatch`
- [ ] 批量操作入参类型为 `List<Model>`

---

## 参考文档

- [ORM-API.md](file:///Users/mac/Documents/coding/6.0/trae/oinone-dev/oinone-backend-tutorials/.trae/source/source/oinone-docs/en/DevManual/Reference/Back-EndFramework/ORM-API.md)（Model、Field、Pops、Service）
- [UX-API.md](file:///Users/mac/Documents/coding/6.0/trae/oinone-dev/oinone-backend-tutorials/.trae/source/oinone-docs/en/DevManual/Reference/Back-EndFramework/UX-API.md)（Menu、View、Widget）
- [model.md](file:///Users/mac/Documents/coding/6.0/trae/oinone-dev/oinone-backend-tutorials/.trae/source/oinone-docs/en/DevManual/Reference/BackendFramework/model.md)（模型设计）
- [DefaultWriteWithFieldApi.java](file:///Users/mac/Documents/coding/6.0/trae/oinone-dev/oinone-backend-tutorials/.trae/source/oinone-pamirs/pamirs-framework/pamirs-framework-orm/src/main/java/pro/shushi/pamirs/framework/orm/DefaultWriteWithFieldApi.java)（重写模型CRUD行为的参考实现）
